package ar.com.sdd.gicapi.api.payment;

import ar.com.sdd.commons.rest.log.ServerFileLogged;
import ar.com.sdd.commons.rest.model.ErrorResult;
import ar.com.sdd.ebfapi.client.model.ActionLink;
import ar.com.sdd.gicapi.client.model.GicDocumentSet;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.parameters.RequestBody;
import io.swagger.v3.oas.annotations.responses.ApiResponse;

import javax.inject.Inject;
import javax.ws.rs.*;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.SecurityContext;

@Path("/gic-api/v1_0/payment")
public class GicPaymentApiEndpoint {

    @Inject private GicPaymentApiService paymentApiService;

    @GET
    @ServerFileLogged
    @Path("/aid/{aid}/owner/{ownerId}/payerCuit/{payerCuit}/payment-methods")
    @Produces("application/json; charset=utf-8")
    @Operation(summary = "Consulta de medios de pagos disponibles a un cliente")
    public Response getPaymentMethods(@HeaderParam("Authorization") String authorization, @PathParam("aid") String applicationId, @PathParam("ownerId") String ownerId,
                                    @PathParam("payerCuit") String payerCuit,
                                    @Context SecurityContext securityContext) {

        return paymentApiService.getPaymentMethods(authorization, applicationId, ownerId, payerCuit, securityContext);
    }

    @GET
    @ServerFileLogged
    @Path("/aid/{aid}/owner/{ownerId}/payerCuit/{payerCuit}/document-set/{screenTemplateName}")
    @Produces("application/json; charset=utf-8")
    public Response getDocumentsSet(@HeaderParam("Authorization") String authorization, @PathParam("aid") String applicationId, @PathParam("ownerId") String ownerId,
                                    @PathParam("payerCuit") String payerCuit,@PathParam("screenTemplateName") String screenTemplateName,
                                    @Context SecurityContext securityContext) {

        return paymentApiService.getDocumentsSet(authorization, applicationId, ownerId, payerCuit, screenTemplateName, securityContext);
    }

    @POST
    @Operation(
            operationId = "postDocumentSetPaymentLink",
            tags = "payment",
            summary = "Creates payment for a set of document and provides a SSO link to it",
            responses = {
                    @ApiResponse(responseCode = "200", description = "Success", content = @Content(schema = @Schema(implementation = ActionLink.class))),
                    @ApiResponse(responseCode = "400", description = "General error", content = @Content(schema = @Schema(implementation = ErrorResult.class))),
                    @ApiResponse(responseCode = "417", description = "Wrong applicationId", content = @Content(schema = @Schema(implementation = ErrorResult.class))),
                    @ApiResponse(responseCode = "418", description = "Wrong owner", content = @Content(schema = @Schema(implementation = ErrorResult.class))),
                    @ApiResponse(responseCode = "419", description = "Invalid PaymentMethod", content = @Content(schema = @Schema(implementation = ErrorResult.class)))
            }
    )
    @ServerFileLogged
    @Path("/aid/{aid}/owner/{ownerId}/document-set/payment-link")
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public Response postDocumentSetPaymentLink(
            @Parameter(description = "Bearer token generated in oAuth2 operation", required = true)
            @HeaderParam("Authorization") String authorization,

            @Parameter(description = "ApplicationId, unique identifier of the application. Value: GIC_EBFAPI", required = true, content = @Content(schema = @Schema(maxLength = 50)))
            @PathParam("aid") String applicationId,

            @Parameter(description = "Company's CUIT", required = true)
            @Schema(maxLength = 255)
            @PathParam("ownerId") String ownerId,

            @RequestBody(description = "Set of documents to create", required = true, content = @Content(schema = @Schema(implementation = GicDocumentSet.class)))
            GicDocumentSet documentSet,

            @Context SecurityContext securityContext) {
        return paymentApiService.postDocumentSetPaymentLink(authorization, applicationId, ownerId, documentSet, securityContext);
    }

    @DELETE
    @ServerFileLogged
    @Path("/aid/{aid}/owner/{ownerId}/document-set/payment-link")
    @Consumes({"application/json"})
    @Produces("application/json; charset=utf-8")
    public Response deleteDocumentSetPaymentLink(@HeaderParam("Authorization") String authorization, @PathParam("aid") String applicationId, @PathParam("ownerId") String ownerId, GicDocumentSet documentSet, @Context SecurityContext securityContext) {
        return paymentApiService.deleteDocumentSetPaymentLink(authorization, applicationId, ownerId, documentSet, securityContext);
    }

    @GET
    @ServerFileLogged
    @Path("/aid/{aid}/owner/{ownerId}/payments")
    @Produces("application/json; charset=utf-8")
    public Response getPayments(@HeaderParam("Authorization") String authorization, @PathParam("aid") String applicationId, @PathParam("ownerId") String ownerId,
                                @QueryParam("fromCreationDate") String fromCreationDate,
                                @QueryParam("toCreationDate") String toCreationDate,
                                @QueryParam("ownerDocRefList") String ownerDocRefList,
                                @QueryParam("legalDocRefList") String legalDocRefList,
                                @QueryParam("offset") Integer offset,
                                @QueryParam("limit") Integer limit,
                                @Context SecurityContext securityContext) {
        return paymentApiService.getPayments(authorization, applicationId, ownerId, fromCreationDate, toCreationDate, ownerDocRefList, legalDocRefList, offset, limit, securityContext);
    }

    @GET
    @ServerFileLogged
    @Path("/aid/{aid}/owner/{ownerId}/payment-status-info/orderNumber/{orderNumber}")
    @Produces("application/json; charset=utf-8")
    public Response getPaymentStatusInfo(@HeaderParam("Authorization") String authorization, @PathParam("aid") String applicationId, @PathParam("ownerId") String ownerId,
                               @PathParam("orderNumber") String orderNumber,
                               @Context SecurityContext securityContext) {
        return paymentApiService.getPaymentStatusInfo(authorization, applicationId, ownerId, orderNumber, securityContext);
    }

}
