package ar.com.sdd.hlcmapi.core;

import ar.com.sdd.commons.rest.core.*;
import ar.com.sdd.commons.rest.model.TokenOAuth2;
import ar.com.sdd.commons.rest.util.RestConnectorUtil;
import ar.com.sdd.hlcmapi.io.InvoicesRequest;
import ar.com.sdd.hlcmapi.io.InvoicesResponse;
import ar.com.sdd.hlcmapi.io.PaymentOrderRequest;
import ar.com.sdd.hlcmapi.io.PaymentOrderResponse;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.Form;
import javax.ws.rs.core.MediaType;

@SuppressWarnings("unused")
public class HlcmApiConnector implements RestSecurityManager {

    private final Logger log = LogManager.getLogger(HlcmApiConnector.class);

    private final RestConnector restConnector;
    private final String username;
    private final String password;
    private final String loginUrl;
    private RequestMethod currentMethod;

    private enum RequestMethod {
        GET_INVOICES, POST_PAYMENT
    }
    public HlcmApiConnector(HlcmApiConnectorContext context) {
        this.username = context.getUsername();
        this.password = context.getPassword();
        this.loginUrl = context.getLoginUrl();

        final String maskedPassword = "X".repeat(password.length() - 4) + password.substring(password.length() - 4);
        log.debug("Creando HlcmConnector para url [{}] con username [{}] y password [{}]", context.getBaseUrl(), username, maskedPassword);

        RestConnectorEnvironment environment = new RestConnectorEnvironment(context.getBaseUrl());
        restConnector = new RestConnector(environment, this);
    }

    private String getAccessToken() throws RestConnectorException {
        log.debug("[getAccessToken] Por recuperar token con username [{}] y password [{}]", username, password.replaceAll(".(?=.{5})", "*"));
        RestConnectorEnvironment environment = new RestConnectorEnvironment(loginUrl);
        RestConnector connector = new RestConnector(environment, new RestSecurityManager() {
            @Override
            public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
                if (builder != null) {
                    builder.header("Authorization", RestConnectorUtil.getBasicAuthHeader(username, password));
                }
                return builder;
            }

            @Override
            public boolean retryOnUnauthorized() {
                return false;
            }

            @Override
            public boolean getDisableHTTPSErrors() {
                return false;
            }
        });
        TokenOAuth2 tokenOAuth2Response = connector.genericPost(new Form("grant_type", "client_credentials"), TokenOAuth2.class, "", MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
        log.debug("[getAccessToken] Token recuperado [{}]", tokenOAuth2Response.getAccessToken());
        return tokenOAuth2Response.getAccessToken();
    }

    public InvoicesResponse getInvoices(String companyCode, String payerCode, String cuit) throws RestConnectorException {
        currentMethod = RequestMethod.GET_INVOICES;
        final String path = "/v1/invoices";
        InvoicesRequest invoicesRequest = new InvoicesRequest(companyCode, payerCode, cuit);
        log.debug("[getInvoices] Request POST getInvoices para el companyCode [{}], payerCode [{}] con path [" + path + "]", companyCode, payerCode);
        return restConnector.genericPost(invoicesRequest, InvoicesResponse.class, path);
    }

    public PaymentOrderResponse postPayments(PaymentOrderRequest paymentOrderRequest) throws RestConnectorException {
        currentMethod = RequestMethod.POST_PAYMENT;
        final String path = "/v1/payment/order";
        log.debug("[postPayments] Request POST payment para {} con path [" + path + "]", paymentOrderRequest);
        return restConnector.genericPost(paymentOrderRequest, PaymentOrderResponse.class, path);
    }

    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        if (builder != null) {
            builder.header("Authorization", getAccessToken());
        }
        return builder;
    }

    @Override
    public boolean retryOnUnauthorized() {
        return false;
    }

    @Override
    public boolean getDisableHTTPSErrors() {
        return false;
    }
}