package ar.com.sdd.hlcmapi.core;

import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import ar.com.sdd.commons.rest.model.TokenOAuth2;
import ar.com.sdd.commons.rest.util.RestConnectorUtil;
import ar.com.sdd.hlcmapi.io.InvoicesRequest;
import ar.com.sdd.hlcmapi.io.InvoicesResponse;
import ar.com.sdd.hlcmapi.io.PaymentOrderRequest;
import ar.com.sdd.hlcmapi.io.PaymentOrderResponse;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.Form;
import javax.ws.rs.core.MediaType;

public class HlcmApiConnector implements RestSecurityManager {

    private final Logger log = LogManager.getLogger(HlcmApiConnector.class);

    private final RestConnector restConnector;
    private final String apiKey;
    private final String username;
    private final String password;
    private final String loginUrl;
    private RequestMethod currentMethod;

    private enum RequestMethod {
        GET_INVOICES, POST_PAYMENT
    }
    public HlcmApiConnector(HlcmApiConnectorContext context) {
        this.apiKey = context.getApiKey();
        this.username = context.getUsername();
        this.password = context.getPassword();
        this.loginUrl = context.getLoginUrl();

        log.debug("Creando HlcmConnector para url [" + context.getBaseUrl() + "] con apiKey [" + apiKey + "]");

        RestConnectorEnvironment environment = new RestConnectorEnvironment(context.getBaseUrl());
        restConnector = new RestConnector(environment, this);
    }

    private String getAccessToken() throws RestConnectorException {
        log.debug("[getAccessToken] Por recuperar token con username [" + username + "] y password [" + password.replaceAll(".(?=.{5})", "*") + "]");
        RestConnectorEnvironment environment = new RestConnectorEnvironment(loginUrl);
        RestConnector connector = new RestConnector(environment, new RestSecurityManager() {
            @Override
            public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
                if (builder != null) {
                    builder.header("Authorization", RestConnectorUtil.getBasicAuthHeader(username, password));
                }
                return builder;
            }

            @Override
            public boolean getDisableHTTPSErrors() {
                return false;
            }
        });
        TokenOAuth2 tokenOAuth2Response = connector.genericPost(new Form("grant_type", "client_credentials"), TokenOAuth2.class, "", MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
        log.debug("[getAccessToken] Token recuperado [" + tokenOAuth2Response.getAccessToken() + "]");
        return tokenOAuth2Response.getAccessToken();
    }

    public InvoicesResponse getInvoices(String companyCode, String payerCode, String cuit) throws RestConnectorException {
        currentMethod = RequestMethod.GET_INVOICES;
        final String path = "/v1/invoices";
        InvoicesRequest invoicesRequest = new InvoicesRequest(companyCode, payerCode, cuit);
        log.debug("[getInvoices] Request POST getInvoices para el companyCode [" + companyCode + "], payerCode [" + payerCode + "] con path [" + path + "]");
        return restConnector.genericPost(invoicesRequest, InvoicesResponse.class, path);
    }

    public PaymentOrderResponse postPayments(PaymentOrderRequest paymentOrderRequest) throws RestConnectorException {
        currentMethod = RequestMethod.POST_PAYMENT;
        final String path = "/v1/payment/order";
        log.debug("[getInvoices] Request POST payment para " + paymentOrderRequest + " con path [" + path + "]");
        return restConnector.genericPost(paymentOrderRequest, PaymentOrderResponse.class, path);
    }

    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        if (builder != null) {
            builder.header("Authorization", getAccessToken());
        }
        return builder;
    }

    @Override
    public boolean getDisableHTTPSErrors() {
        return false;
    }
}