package ar.com.sdd.interbanking.core;

import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import ar.com.sdd.commons.rest.model.TokenOAuth2;
import ar.com.sdd.commons.rest.util.QueryBuilder;
import ar.com.sdd.interbanking.io.recaudador.DeudaRequest;
import ar.com.sdd.interbanking.io.recaudador.DeudaResponse;
import ar.com.sdd.interbanking.io.recaudador.RendicionesResponse;
import ar.com.sdd.interbanking.io.recaudador.TransferenciasResponse;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.Form;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.AbstractMap;
import java.util.Date;
import java.util.Map;

/*
 * Ver documentacion en InterbankingApiConnector
 */

@SuppressWarnings("unused")
public class InterbankingApiKeyConnector extends InterbankingApiBaseConnector {

    @Override
    public ConnectorClass getConnectorClass() { return ConnectorClass.APIKEY; }

    private static final Logger log = LogManager.getLogger(InterbankingApiKeyConnector.class);

    private RestConnectorEnvironment tokenEnvironment;
    private RestConnectorEnvironment environment;
    private RestConnector restConnector;
    private String grantType;
    private String username;
    private String password;
    private String clientId; //este es el clienteId asociado a una apiKey
    private String scope;  // como puede tener una coma y molesta, usamos __COMA__

    public InterbankingApiKeyConnector(InterbankingApiKeyConnectorContext interbankingApiConnectorContext) {
        this.tokenEnvironment = new RestConnectorEnvironment(
                interbankingApiConnectorContext.getAuthBaseUrl(),
                null,
                null,
                interbankingApiConnectorContext.getTrustStorePath(),
                interbankingApiConnectorContext.getTrustStorePassword());
        this.environment = new RestConnectorEnvironment(interbankingApiConnectorContext.getBaseUrl(),
                null,
                null,
                interbankingApiConnectorContext.getTrustStorePath(),
                interbankingApiConnectorContext.getTrustStorePassword());
        this.restConnector = new RestConnector(environment, this);
        this.grantType = interbankingApiConnectorContext.getGrantType();
        this.username = interbankingApiConnectorContext.getUsername();
        this.password = interbankingApiConnectorContext.getPassword();
        this.clientId = interbankingApiConnectorContext.getClientId();
        this.scope = interbankingApiConnectorContext.getScope();
        if (scope!=null) {
            scope = scope.replace("__COMA__", ",");
        }

        log.debug("Creando connector InterbankingApiKey con ["
                + "tokenBaseUrl='" +( (this.tokenEnvironment==null)?"null":this.tokenEnvironment.baseUrl) + "',"
                + "baseUrl='" + this.environment.baseUrl + "',"
                + "grantType='" + this.grantType + "',"
                + "grantType='" + this.grantType + "',"
                + "username='" + this.username + "',"
                + "password='" + this.password + "',"
                + "clientId='" + this.clientId + "',"
                + "scope='" + this.scope + "']");
    }

    //Estas API usan conexion via apiKey
    public DeudaResponse postDeuda(DeudaRequest deudaRequest) throws RestConnectorException {
        log.debug("[postDeuda] Request POST deuda con " + deudaRequest);
        DeudaResponse response = restConnector.genericPost(deudaRequest, DeudaResponse.class, "/api/recaudador/v1.0.0/deudas/", "application/json", "application/json;charset=UTF-8");
        log.debug("[postDeuda] Response POST deuda [" + response + "]");
        return response;
    }

    public DeudaResponse deleteDeuda(Long identificadorDeuda) throws RestConnectorException{
        log.debug("[deleteDeuda] Request DELETE deuda con identificadorDeuda [" + identificadorDeuda + "]");
        DeudaResponse response = restConnector.genericDelete(null, DeudaResponse.class, "/api/recaudador/v1.0.0/deudas/{identificadorDeuda}", Map.ofEntries(new AbstractMap.SimpleEntry<String, Object>("identificadorDeuda", identificadorDeuda)));
        log.debug("[deleteDeuda] Response DELETE deuda [" + response + "]");
        return response;
    }

    public DeudaResponse getDeuda(Long identificadorDeuda) throws RestConnectorException {
        log.debug("[getDeuda] Request GET deuda con identificadorDeuda [" + identificadorDeuda + "]");
        DeudaResponse response = restConnector.genericGet(null, DeudaResponse.class, "/api/recaudador/v1.0.0/deudas/{identificadorDeuda}", "identificadorDeuda", String.valueOf(identificadorDeuda));
        log.debug("[getDeuda] Response GET deuda [" + response + "]");
        return response;
    }

    public RendicionesResponse getRendiciones(Date fecha, String moneda, Integer registrosPorPagina, Integer indiceDePagina) throws RestConnectorException {
        QueryBuilder builder = QueryBuilder.Builder().path("/api/recaudador/v1.0.0/deudas/rendiciones");
        if (fecha != null) {
            builder.add("fecha", new SimpleDateFormat("yyyy-MM-dd").format(fecha));
        } else {
            log.warn("La fecha es obligatoria para solicitar las rendiciones a IB");
            return null;
        }

        if (moneda != null) {
            if (moneda.equals("ARS") || moneda.equals("USD")) {
                builder.add("moneda", moneda);
            } else {
                log.warn("La moneda [" + moneda + "] no es valida, solamente se acepta ARS o USD. Se realiza la busqueda sin filtrar por moneda.");
            }
        }

        if (registrosPorPagina != null) {
            if (registrosPorPagina > 0 && registrosPorPagina <= 1000) {
                builder.add("tamano", registrosPorPagina);
            } else {
                log.warn("La cantidad de registros por pagina debe ser entre 1 y 1000");
            }
        }

        if (indiceDePagina != null) {
            builder.add("pagina", indiceDePagina);
        }

        String path = builder.build();

        log.debug("[getRendiciones] Request GET rendiciones con fecha [" + fecha + "], moneda [" + moneda + "], registrosPorPagina [" + registrosPorPagina + "], indiceDePagina [" + indiceDePagina + "]");
        RendicionesResponse response = restConnector.genericGet(null, RendicionesResponse.class, path);
        log.debug("[getRendiciones] Response GET rendiciones [" + response + "]");
        return response;
    }

    public TransferenciasResponse postTransferencias(String cuitVendedor, String cuitPagador, BigDecimal importeDesde, BigDecimal importeHasta, Date fechaDesde, Date fechaHasta, String nroComprobanteDesde, String nroComprobanteHasta, String idDeudaDesde, String idDeudaHasta, String estado) throws RestConnectorException {
        QueryBuilder builder = QueryBuilder.Builder().path("/api/recaudador/v1.0.0/consultas/transferencias");
        if (cuitVendedor != null) {
            builder.add("cuitVendedor", cuitVendedor);
        }

        if (cuitPagador != null) {
            builder.add("cuitPagador", cuitPagador);
        }

        if (importeDesde != null) {
            builder.add("importeDesde", importeDesde);
        }

        if (importeHasta != null) {
            builder.add("importeHasta", importeHasta);
        }

        if (fechaDesde != null) {
            builder.add("fechaDesde", new SimpleDateFormat("yyyy-MM-dd").format(fechaDesde));
        }

        if (fechaHasta != null) {
            builder.add("fechaHasta", new SimpleDateFormat("yyyy-MM-dd").format(fechaHasta));
        }

        if (nroComprobanteDesde != null) {
            builder.add("nroComprobanteDesde", nroComprobanteDesde);
        }

        if (nroComprobanteHasta != null) {
            builder.add("nroComprobanteHasta", nroComprobanteHasta);
        }

        if (idDeudaDesde != null) {
            builder.add("idDeudaDesde", idDeudaDesde);
        }

        if (idDeudaHasta != null) {
            builder.add("idDeudaHasta", idDeudaHasta);
        }

        if (estado != null) {
            builder.add("estado", estado);
        }

        log.debug("[postTransferencias] Request POST transferencias con "
                + "cuitVendedor [" + cuitVendedor + "],"
                + "cuitPagador [" + cuitPagador + "],"
                + "importeDesde [" + importeDesde + "],"
                + "importeHasta [" + importeHasta + "],"
                + "fechaDesde [" + fechaDesde + "],"
                + "fechaHasta [" + fechaHasta + "],"
                + "nroComprobanteDesde [" + nroComprobanteDesde + "],"
                + "nroComprobanteHasta [" + nroComprobanteHasta + "],"
                + "idDeudaDesde [" + idDeudaDesde + "],"
                + "idDeudaHasta [" + idDeudaHasta + "],"
                + "estado [" + estado + "]");
        TransferenciasResponse response = restConnector.genericPost(null, TransferenciasResponse.class, builder.build());
        log.debug("[postTransferencias] Response POST transferencias [" + response + "]");
        return response;
    }


    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        TokenOAuth2 token = getToken();
        log.trace("[addHeaders] Token recuperado [" + token + "]");
        builder.header("Authorization", "Bearer " + token.getAccessToken());
        return builder;
    }

    @Override
    public boolean getDisableHTTPSErrors() {
        return false;
    }
    @Override
    public boolean retryOnUnauthorized() {
        return false;
    }


    private TokenOAuth2 getToken() throws RestConnectorException {
        Form form = new Form();
        form.param("grant_type", this.grantType);
        form.param("username", this.username);
        form.param("password", this.password);
        form.param("client_id", this.clientId);

        log.debug("[getToken] Por recuperar token con clientId [" + this.clientId + "], username [" + this.username + "], scope [" + this.scope + "]");
        RestConnector tokenConnector = new RestConnector(this.tokenEnvironment, new TokenSecurityManager(environment));
        TokenOAuth2 tokenOAuth2Response = tokenConnector.genericPost(form, TokenOAuth2.class, "/cas/oauth2.0/token?scope=" + this.scope, "application/json", "application/x-www-form-urlencoded");
        log.debug("[getToken] Token recuperado [" + tokenOAuth2Response + "]");
        return tokenOAuth2Response;
    }

    public static class TokenSecurityManager implements RestSecurityManager {

        RestConnectorEnvironment environment;

        TokenSecurityManager(RestConnectorEnvironment environment) {
            this.environment = environment;
        }

        @Override
        public Invocation.Builder addHeaders(Invocation.Builder builder) {
            builder.header("service", environment.baseUrl );
            return builder;
        }

        @Override
        public boolean getDisableHTTPSErrors() {
            return false;
        }
        @Override
        public boolean retryOnUnauthorized() {
            return false;
        }

    }
}
