package ar.com.sdd.interbanking.core;

import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import ar.com.sdd.commons.rest.model.TokenOAuth2;
import ar.com.sdd.commons.rest.util.QueryBuilder;
import ar.com.sdd.interbanking.io.financiero.*;
import ar.com.sdd.interbanking.model.financiero.ErrorMessage;
import ar.com.sdd.interbanking.util.StringUtil;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.Form;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/*
*  IB tiene tres modelos de conexion, a medidda que fue evolucionando
*  1) apiKey       : version inicial, se usa para publicar deuda y consultar pagos
*  2) certificado  : version primera de MBO usandoautenticacion x certificados
*  3) token       : version actual de MBO usando autenticacion x token
*
*  La 1 esta se usa desde InterbankingApiKey
*      Se usa para publicar deuda y consultar pagos para el modelo de publicacion de deuda
*      Se uso temporalmente para la implementacion del modelo TPNAPI, que en vez
*      de generar un pago BtoB, directamente, generaba una publiacciaon de deuda y luego
*      remite al cliente a pagarla
*      Este modelo lo expejamos en ar.com.sdd.ebfapi.client.model.contrato.inba para poder exponer los mismos servicios
*      que IB, para que los use CVSA
*      El model que usamos como cliente de IB, ahora esta bajo recaudaror.*
*
*  La 2 desde InterbankingApiCertificateConnector
*      El modelo asociado esta bajo financiero_v1
*
*  La 3 es la que deberia sobrevivir a largo plazo
*      El modelo asociado esta bajco financiero
*
*
*
*
*/

@SuppressWarnings("unused")
public class InterbankingApiConnector implements RestSecurityManager {

    private static final Logger log = LogManager.getLogger(InterbankingApiConnector.class);

    private InterbankingApiConnectorContext context;
    private RestConnectorEnvironment tokenEnvironment;
    private RestConnectorEnvironment environment;
    private RestConnector restConnector;

    public InterbankingApiConnector(InterbankingApiConnectorContext interbankingApiConnectorContext) {
        this.context = interbankingApiConnectorContext;
        this.environment = new RestConnectorEnvironment(
                context.getBaseUrl(),
                null,
                null,
                null,
                null);
        this.tokenEnvironment = new RestConnectorEnvironment(
                context.getTokenBaseUrl(),
                null,
                null,
                null,
                null);
        this.restConnector = new RestConnector(environment, this);

        log.debug("Creando connector InterbankingApiConnector con ["
                + "baseUrl='" + this.environment.baseUrl + "',"
                + "clientId='" + this.context.getClientId()+ ""
                + "']");
    }




    //accountNumber es optativo
    public AccountsResponse getAccounts(String clientCode, AccountsRequest accountsRequest) throws RestConnectorException {
        log.debug("[getAccounts] Request GET con clientCode ["+clientCode+"]");

        String path="/accounts";
        Map<String, Object> templateParameters = new HashMap<>();

        if (!StringUtil.isEmptyNull(accountsRequest.getAccountNumber())){
            path+="/{accountNumber}";
            templateParameters.put("accountNumber", accountsRequest.getAccountNumber());
        }

        QueryBuilder builder = QueryBuilder.Builder().path(path);

        StringBuffer requestParamsString = new StringBuffer();
        addParameter(builder, requestParamsString, "customer-id" ,clientCode);
        addParameter(builder, requestParamsString, "bank-number" ,accountsRequest.getBankNumber());
        addParameter(builder, requestParamsString, "account-type",accountsRequest.getAccountType());
        addParameter(builder, requestParamsString, "currency"    ,accountsRequest.getCurrency());
        addParameter(builder, requestParamsString, "limit"       ,accountsRequest.getLimit());
        addParameter(builder, requestParamsString, "page"        ,accountsRequest.getPage());

        log.debug("[getAccounts] Request GET con " + requestParamsString);
        AccountsResponse response = restConnector.genericGet(null, AccountsResponse.class, ErrorMessage.class,builder.build(),templateParameters);
        log.debug("[getAccounts] Response GET [" + response + "]");
        return response;
    }

    public BalanceResponse getBalance(String clientCode, BalanceRequest  balanceRequest) throws RestConnectorException {
        log.debug("[getBalance] Request GET con clientCode ["+clientCode+"]");
        if (balanceRequest == null) {
            log.warn("[getBalance] No se realiza la consulta de saldos porque balanceRequest es null");
            return null;
        }

        String path="/accounts/balances";
        Map<String, Object> templateParameters = new HashMap<>();

        QueryBuilder builder = QueryBuilder.Builder().path(path);

        StringBuffer requestParamsString = new StringBuffer();
        addParameter(builder, requestParamsString, "customer-id"    ,clientCode);
        addParameter(builder, requestParamsString, "bank-number"    ,balanceRequest.getBankNumber());
        addParameter(builder, requestParamsString, "account-number" ,balanceRequest.getAccountNumber());
        addParameter(builder, requestParamsString, "account-type"   ,balanceRequest.getAccountType());
        addParameter(builder, requestParamsString, "currency"       ,balanceRequest.getCurrency());
        addParameter(builder, requestParamsString, "date-since"     ,balanceRequest.getDateSince());
        addParameter(builder, requestParamsString, "date-until"     ,balanceRequest.getDateUntil());
        addParameter(builder, requestParamsString, "limit"          ,balanceRequest.getLimit());
        addParameter(builder, requestParamsString, "page"           ,balanceRequest.getPage());


        log.debug("[getBalance] Request GET con " + requestParamsString);
        BalanceResponse response = restConnector.genericGet(null, BalanceResponse.class, ErrorMessage.class,builder.build(),templateParameters);
        log.debug("[getBalance] Response GET [" + response + "]");
        return response;
    }

    public MovementsResponse getMovements(String clientCode, MovementsRequest  movementsRequest) throws RestConnectorException {
        log.debug("[getMovements] Request GET con clientCode ["+clientCode+"]");
        if (movementsRequest == null) {
            log.warn("[getMovements] No se realiza la consulta de saldos porque movementsRequest es null");
            return null;
        }

        String path="/accounts/{account-number}/movements/{movement-type}";
        Map<String, Object> templateParameters = new HashMap<>();
        templateParameters.put("account-number", movementsRequest.getAccountNumber());
        templateParameters.put("movement-type", movementsRequest.getMovementType());

        QueryBuilder builder = QueryBuilder.Builder().path(path);

        StringBuffer requestParamsString = new StringBuffer();

        addParameter(builder, requestParamsString, "customer-id"    ,clientCode);
        addParameter(builder, requestParamsString, "bank-number"    ,movementsRequest.getBankNumber());
        addParameter(builder, requestParamsString, "account-number" ,movementsRequest.getAccountNumber());
        addParameter(builder, requestParamsString, "account-type"   ,movementsRequest.getAccountType());
        addParameter(builder, requestParamsString, "currency"       ,movementsRequest.getCurrency());
        addParameter(builder, requestParamsString, "date-since"     ,movementsRequest.getDateSince());
        addParameter(builder, requestParamsString, "date-until"     ,movementsRequest.getDateUntil());
        addParameter(builder, requestParamsString, "limit"          ,movementsRequest.getLimit());
        addParameter(builder, requestParamsString, "page"           ,movementsRequest.getPage());


        log.debug("[getMovements] Request GET con " + requestParamsString);
        MovementsResponse response = restConnector.genericGet(null, MovementsResponse.class, ErrorMessage.class,builder.build(),templateParameters);
        log.debug("[getMovements] Response GET [" + response + "]");
        return response;
    }


    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        TokenOAuth2 token = getToken();
        //log.trace("[addHeaders] Token recuperado [" + token + "]");
        builder.header("Authorization", "Bearer " + token.getAccessToken());
        builder.header("client_id", context.getClientId());
        return builder;
    }


    private TokenOAuth2 getToken() throws RestConnectorException {
        Form form = new Form();
        form.param("grant_type", "client_credentials");
        form.param("client_secret", context.getClientSecret());
        form.param("client_id", context.getClientId());

        log.trace("[getToken] Por recuperar token con clientId [" + context.getClientId() + "], scope [" + context.getScope() + "]");
        RestConnector tokenConnector = new RestConnector(this.tokenEnvironment, new TokenSecurityManager(environment, context.getService()));
        TokenOAuth2 tokenOAuth2Response = tokenConnector.genericPost(form, TokenOAuth2.class, "/cas/oidc/accessToken?scope=" + context.getScope(), "application/json", "application/x-www-form-urlencoded");
        log.debug("[getToken] Token recuperado [" + tokenOAuth2Response + "]");
        return tokenOAuth2Response;
    }

    public static class TokenSecurityManager implements RestSecurityManager {

        RestConnectorEnvironment environment;
        String   service ;

        TokenSecurityManager(RestConnectorEnvironment environment, String service) {
            this.environment = environment;
            this.service = service;
        }

        @Override
        public Invocation.Builder addHeaders(Invocation.Builder builder) {
            builder.header("service", this.service);
            return builder;
        }

        @Override
        public boolean getDisableHTTPSErrors() {
            return false;
        }

    }

    @Override
    public boolean getDisableHTTPSErrors() {
        return false;
    }

    private void addParameter(QueryBuilder builder, StringBuffer parameterString, String parameterName, String parameterValue) {
        if (!StringUtil.isEmptyNull(parameterValue)) {
            builder.add(parameterName, parameterValue);
            parameterString.append(" ").append(parameterName).append(" [").append(parameterValue).append("]");
        }
    }
    private void addParameter(QueryBuilder builder, StringBuffer parameterString, String parameterName, Date parameterValue) {
        if (parameterValue!=null) {
            String sParameterValue = new SimpleDateFormat("yyyy-MM-dd").format(parameterValue);
            builder.add(parameterName, sParameterValue);
            parameterString.append(" ").append(parameterName).append(" [").append(sParameterValue).append("]");
        }
    }

}
