package ar.com.sdd.macroapi.core;

import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import ar.com.sdd.commons.rest.model.TokenOAuth2;
import ar.com.sdd.commons.rest.util.RestConnectorUtil;
import ar.com.sdd.commons.util.SimpleCache;
import ar.com.sdd.commons.util.SimpleCacheManager;
import ar.com.sdd.commons.util.StringUtil;
import ar.com.sdd.macroapi.io.CommonError;
import ar.com.sdd.macroapi.io.GetOperationIdResponse;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.Form;
import javax.ws.rs.core.MediaType;

@SuppressWarnings("unused")
public class MacroApiConnector implements RestSecurityManager {

    private final static Logger log = LogManager.getLogger(MacroApiConnector.class);

    private final RestConnector connector;
    private final String username;
    private final String password;
    private boolean doingLogin = false;

    private String xRequestId;

    public MacroApiConnector(MacroApiConnectorContext context) {
        this.username = context.getUsername();
        this.password = context.getPassword();

        RestConnectorEnvironment environment = new RestConnectorEnvironment(context.getBaseUrl());
        connector = new RestConnector(environment, this);
    }

    public GetOperationIdResponse getOperationId(String requestId) throws RestConnectorException{
        final String path = "/v1/agro-loans/operations-id";
        log.debug("[getOperationId] Request GET por pedir el id de operacion para el requestId [{}] path [{}]", requestId, path);
        // Lo lleno para usarlo en el header
        xRequestId = requestId;
        GetOperationIdResponse response = connector.genericGet(null, GetOperationIdResponse.class, CommonError.class, path);
        // Lo borro para evitar incluirlo en otros request
        xRequestId = null;
        return response;
    }

    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        if (builder != null) {
            if (doingLogin) {
                builder.header("Authorization", RestConnectorUtil.getBasicAuthHeader(username, password));
            } else {
                builder.header("Authorization", "Bearer " + getAccessToken(false));

                // Se llena en la consulta de operation id by requestId
                if (StringUtil.isNotEmpty(xRequestId)) {
                    builder.header("x-request-id", xRequestId);
                }
            }
        }
        return builder;
    }

    @Override
    public boolean retryOnUnauthorized() {
        try {
            getAccessToken(true);
            return true;
        } catch (RestConnectorException e) {
            log.error(e);
            return false;
        }
    }

    @Override
    public boolean getDisableHTTPSErrors() {
        return false;
    }

    public String getAccessToken(boolean force) throws RestConnectorException {
        SimpleCache cache = SimpleCacheManager.getInstance().getCache(MacroApiConnector.class.getName());
        String cacheKey = SimpleCacheManager.buildKey("accessToken");
        String accessToken = (String) cache.get(cacheKey);
        if (force || accessToken == null) {
            Form form = new Form();
            form.param("scope", "openid");
            form.param("grant_type", "client_credentials");

            log.debug("[getAccessToken] Por obtener token con username [{}] y password [{}]", username, password);

            doingLogin = true;
            TokenOAuth2 tokenOAuth2Response = connector.genericPost(form, TokenOAuth2.class, "/v1/oauth/token", MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
            doingLogin = false;

            accessToken = tokenOAuth2Response.getAccessToken();
            log.debug("[getAccessToken] Token recuperado [{}]. Expira en [{}] segs. Lo agrego al a cache", accessToken, tokenOAuth2Response.getExpiresIn());
            cache.put(cacheKey, accessToken, tokenOAuth2Response.getExpiresIn() - 10);
        }

        return accessToken;
    }
}
