package ar.com.sdd.macroapi.core;

import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import ar.com.sdd.commons.rest.model.TokenOAuth2;
import ar.com.sdd.commons.rest.util.RestConnectorUtil;
import ar.com.sdd.commons.util.DateUtil;
import ar.com.sdd.commons.util.SimpleCache;
import ar.com.sdd.commons.util.SimpleCacheManager;
import ar.com.sdd.commons.util.StringUtil;
import ar.com.sdd.macroapi.io.*;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.Form;
import javax.ws.rs.core.MediaType;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

@SuppressWarnings("unused")
public class MacroApiConnector implements RestSecurityManager {

    private final static Logger log = LogManager.getLogger(MacroApiConnector.class);

    private final RestConnector connector;
    private final String username;
    private final String password;
    private boolean doingLogin = false;

    private String xRequestId;

    public enum FilterOptions {
        ALL("ALL"),
        DISTRIBUTOR("DISTRIBUTOR"),
        AGRO_SUPPLIER("AGRO-SUPPLIER");

        private final String value;

        FilterOptions(String value) {
            this.value = value;
        }
    }

    public enum Status {
        ACEPTADA,
        ERROR,
        EXPIRADA,
        LIQUIDADA,
        PENDIENTE,
        RECHAZADA,
        REVERSADA;
    }

    public MacroApiConnector(MacroApiConnectorContext context) {
        this.username = context.getUsername();
        this.password = context.getPassword();

        RestConnectorEnvironment environment = new RestConnectorEnvironment(context.getBaseUrl());
        connector = new RestConnector(environment, this);
    }

    public GetOperationIdResponse getOperationId(String requestId) throws RestConnectorException {
        final String path = "/v1/agro-loans/operations-id";
        log.debug("[getOperationId] Request GET por pedir el id de operacion para el requestId [{}] path [{}]", requestId, path);
        // Lo lleno para usarlo en el header
        xRequestId = requestId;
        GetOperationIdResponse response = connector.genericGet(null, GetOperationIdResponse.class, CommonError.class, path);
        // Lo borro para evitar incluirlo en otros request
        xRequestId = null;
        return response;
    }

    public GetApplicationsResponse getApplications(String requestId, String operationId, String customerTaxPayerNumber, String distributorTaxPayerNumber, FilterOptions filterOptions, List<Long> productCategoriesIds, Status statusId, Date dateFrom, Date dateTo, String orderBy, String order, Integer page, Integer recordsNumber) throws RestConnectorException {
        String path = "/v1/agro-loans/applications?operation-id=" + operationId;
        if (StringUtil.isNotEmpty(customerTaxPayerNumber)) path += "&customer-taxpayer-number=" + customerTaxPayerNumber;
        if (StringUtil.isNotEmpty(distributorTaxPayerNumber)) path += "&distributor-taxpayer-number=" + distributorTaxPayerNumber;
        if (filterOptions != null) path += "&filter-options=" + filterOptions.value;
        if (CollectionUtils.isNotEmpty(productCategoriesIds)) path += "&product-categories-ids=" + productCategoriesIds.stream().map(String::valueOf).collect(Collectors.joining(","));
        if (statusId != null) path += "&status-id=" + statusId.name();
        if (dateFrom != null) path += "&date-from=" + DateUtil.formatJsonDate(dateFrom);
        if (dateTo != null) path += "&date-to=" + DateUtil.formatJsonDate(dateTo);
        if (StringUtil.isNotEmpty(orderBy)) path += "&order-by=" + orderBy;
        if (StringUtil.isNotEmpty(order)) path += "&order=" + order;
        if (page != null) path += "&page=" + page;
        if (recordsNumber != null) path += "&records-number=" + recordsNumber;

        log.debug("[getApplications] Request GET por pedir aplicaciones con path [{}]", path);
        // Lo lleno para usarlo en el header
        xRequestId = requestId;
        GetApplicationsResponse response = connector.genericGet(null, GetApplicationsResponse.class, CommonError.class, path);
        // Lo borro para evitar incluirlo en otros request
        xRequestId = null;
        return response;
    }

    public GetProductsResponse getProducts(String requestId, String customerKey, String operationId) throws RestConnectorException {
        final String path = "/v1/agro-loans/products/customers/" + customerKey + "?operation-id=" + operationId;
        log.debug("[getProducts] Request GET por pedir productos para el cliente [{}] con path [{}]", customerKey, path);
        // Lo lleno para usarlo en el header
        xRequestId = requestId;
        GetProductsResponse response = connector.genericGet(null, GetProductsResponse.class, CommonError.class, path);
        // Lo borro para evitar incluirlo en otros request
        xRequestId = null;
        return response;
    }

    public GetCustomerAccountsResponse getCustomerAccounts(String requestId, String customerTaxPayerNumber, String operationId, String distributorTaxPayerNumber, Long currencyId) throws RestConnectorException {
        String path = "/v1/agro-loans/accounts/taxpayer-numbers/" + customerTaxPayerNumber + "?operation-id=" + operationId + "&currency-id=" + currencyId;
        if (StringUtil.isNotEmpty(distributorTaxPayerNumber)) path += "&distributor-taxpayer-number=" + distributorTaxPayerNumber;
        log.debug("[getCustomerAccounts] Request GET por pedir las cuentas para el cliente [{}] con path [{}]", customerTaxPayerNumber, path);
        // Lo lleno para usarlo en el header
        xRequestId = requestId;
        GetCustomerAccountsResponse response = connector.genericGet(null, GetCustomerAccountsResponse.class, CommonError.class, path);
        // Lo borro para evitar incluirlo en otros request
        xRequestId = null;
        return response;
    }

    public PostAssessmentsResponse postAssessments(String requestId, String customerKey, String operationId, Long accountId, String userIdentity) throws RestConnectorException {
        final String path = "/v1/agro-loans/assessments/customers/" + customerKey + "/availability?operation-id=" + operationId + "&account-id=" + accountId + "&user-identity=" + userIdentity;
        log.debug("[postAssessments] Request POST por crear un assessment para el cliente [{}] con path [{}]", customerKey, path);
        // Lo lleno para usarlo en el header
        xRequestId = requestId;
        PostAssessmentsResponse response = connector.genericGet(null, PostAssessmentsResponse.class, CommonError.class, path);
        // Lo borro para evitar incluirlo en otros request
        xRequestId = null;
        return response;
    }

    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        if (builder != null) {
            if (doingLogin) {
                builder.header("Authorization", RestConnectorUtil.getBasicAuthHeader(username, password));
            } else {
                builder.header("Authorization", "Bearer " + getAccessToken(false));

                // Se llena en la consulta de operation id by requestId
                if (StringUtil.isNotEmpty(xRequestId)) {
                    builder.header("x-request-id", xRequestId);
                }
            }
        }
        return builder;
    }

    @Override
    public boolean retryOnUnauthorized() {
        try {
            getAccessToken(true);
            return true;
        } catch (RestConnectorException e) {
            log.error(e);
            return false;
        }
    }

    @Override
    public boolean getDisableHTTPSErrors() {
        return false;
    }

    public String getAccessToken(boolean force) throws RestConnectorException {
        SimpleCache cache = SimpleCacheManager.getInstance().getCache(MacroApiConnector.class.getName());
        String cacheKey = SimpleCacheManager.buildKey("accessToken");
        String accessToken = (String) cache.get(cacheKey);
        if (force || accessToken == null) {
            Form form = new Form();
            form.param("scope", "openid");
            form.param("grant_type", "client_credentials");

            log.debug("[getAccessToken] Por obtener token con username [{}] y password [{}]", username, password);

            doingLogin = true;
            TokenOAuth2 tokenOAuth2Response = connector.genericPost(form, TokenOAuth2.class, "/v1/oauth/token", MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
            doingLogin = false;

            accessToken = tokenOAuth2Response.getAccessToken();
            log.debug("[getAccessToken] Token recuperado [{}]. Expira en [{}] segs. Lo agrego al a cache", accessToken, tokenOAuth2Response.getExpiresIn());
            cache.put(cacheKey, accessToken, tokenOAuth2Response.getExpiresIn() - 10);
        }

        return accessToken;
    }
}
