package ar.com.sdd.macroapi.core;

import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import ar.com.sdd.commons.rest.model.TokenOAuth2;
import ar.com.sdd.commons.rest.util.RestConnectorUtil;
import ar.com.sdd.commons.util.DateUtil;
import ar.com.sdd.commons.util.SimpleCache;
import ar.com.sdd.commons.util.SimpleCacheManager;
import ar.com.sdd.commons.util.StringUtil;
import ar.com.sdd.macroapi.io.*;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.Form;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

@SuppressWarnings("unused")
public class MacroApiConnector implements RestSecurityManager {

    private final static Logger log = LogManager.getLogger(MacroApiConnector.class);

    private final RestConnector connector;
    private final String username;
    private final String password;
    private boolean doingLogin = false;

    private final boolean isSandbox;

    public enum FilterOptions {
        ALL("ALL"),
        DISTRIBUTOR("DISTRIBUTOR"),
        AGRO_SUPPLIER("AGRO-SUPPLIER");

        private final String value;

        FilterOptions(String value) {
            this.value = value;
        }
    }

    public enum Status {
        ACEPTADA,
        ERROR,
        EXPIRADA,
        LIQUIDADA,
        PENDIENTE,
        RECHAZADA,
        REVERSADA;
    }

    public MacroApiConnector(MacroApiConnectorContext context) {
        this.username = context.getUsername();
        this.password = context.getPassword();
        this.isSandbox = context.isSandbox();

        RestConnectorEnvironment environment = new RestConnectorEnvironment(context.getBaseUrl());
        connector = new RestConnector(environment, this);
    }

    public GetOperationIdResponse getOperationId() throws RestConnectorException {
        final String path = "/v1/agro-loans/operations-id";
        log.debug("[getOperationId] Request GET por pedir un id de operacion con path [{}]", path);
        Pair<GetOperationIdResponse, MultivaluedMap<String,Object>> getOperationIdResponseWithHeaders = connector.genericGetWithResponseHeader(null, GetOperationIdResponse.class, CommonError.class, path);
        MultivaluedMap<String,Object> responseHeaders = getOperationIdResponseWithHeaders.getRight();
        if (responseHeaders != null && responseHeaders.containsKey("x-request-id")) {
            log.debug("[getOperationId] Response x-request-id: {}", responseHeaders.get("x-request-id").get(0));
        }
        return getOperationIdResponseWithHeaders.getLeft();
    }

    public GetCustomerAccountsResponse getCustomerAccounts(String operationId, String customerTaxPayerNumber, String distributorTaxPayerNumber, Long currencyId) throws RestConnectorException {
        String path = "/v1/agro-loans/accounts/taxpayer-numbers/" + customerTaxPayerNumber + "?operation-id=" + operationId + "&currency-id=" + currencyId;
        if (StringUtil.isNotEmpty(distributorTaxPayerNumber)) path += "&distributor-taxpayer-number=" + distributorTaxPayerNumber;
        log.debug("[getCustomerAccounts] Request GET por pedir las cuentas del cliente con path [{}]", path);
        Pair<GetCustomerAccountsResponse, MultivaluedMap<String,Object>> getCustomerAccountsResponseWithHeaders = connector.genericGetWithResponseHeader(null, GetCustomerAccountsResponse.class, CommonError.class, path);
        MultivaluedMap<String,Object> responseHeaders = getCustomerAccountsResponseWithHeaders.getRight();
        if (responseHeaders != null && responseHeaders.containsKey("x-request-id")) {
            log.debug("[getCustomerAccounts] Response x-request-id: {}", responseHeaders.get("x-request-id").get(0));
        }
        return getCustomerAccountsResponseWithHeaders.getLeft();
    }

    public PostAssessmentsResponse postAssessmentsAvailability(String operationId, String customerKey, Long accountId, String userIdentity) throws RestConnectorException {
        final String path = "/v1/agro-loans/assessments/customers/" + customerKey + "/availability?operation-id=" + operationId + "&account-id=" + accountId + "&user-identity=" + userIdentity;
        log.debug("[postAssessments] Request POST por crear un assessment con path [{}]", path);
        Pair<PostAssessmentsResponse, MultivaluedMap<String,Object>> postAssessmentResponseWithHeaders = connector.genericPostWithResponseHeader(null, PostAssessmentsResponse.class, CommonError.class, path, null);
        MultivaluedMap<String,Object> responseHeaders = postAssessmentResponseWithHeaders.getRight();
        if (responseHeaders != null && responseHeaders.containsKey("x-request-id")) {
            log.debug("[postAssessments] Response x-request-id: {}", responseHeaders.get("x-request-id").get(0));
        }
        return postAssessmentResponseWithHeaders.getLeft();
    }

    public GetProductsResponse getProducts(String operationId, String customerKey) throws RestConnectorException {
        final String path = "/v1/agro-loans/products/customers/" + customerKey + "?operation-id=" + operationId;
        log.debug("[getProducts] Request GET por pedir productos con path [{}]", path);
        Pair<GetProductsResponse, MultivaluedMap<String,Object>> getProductsResponseWithHeaders = connector.genericGetWithResponseHeader(null, GetProductsResponse.class, CommonError.class, path);
        MultivaluedMap<String,Object> responseHeaders = getProductsResponseWithHeaders.getRight();
        if (responseHeaders != null && responseHeaders.containsKey("x-request-id")) {
            log.debug("[getProducts] Response x-request-id: {}", responseHeaders.get("x-request-id").get(0));
        }
        return getProductsResponseWithHeaders.getLeft();
    }

    public GetSupplierAccountsResponse getSupplierAccounts(String operationId, Long currencyId) throws RestConnectorException {
        final String path = "/v1/agro-loans/supplier/accounts?operation-id=" + operationId + "&currency-id=" + currencyId;
        log.debug("[getSupplierAccounts] Request GET por pedir cuentas de la insumera (biller) con path [{}]", path);
        Pair<GetSupplierAccountsResponse, MultivaluedMap<String,Object>> getSupplierAccountsResponseWithHeaders = connector.genericGetWithResponseHeader(null, GetSupplierAccountsResponse.class, CommonError.class, path);
        MultivaluedMap<String,Object> responseHeaders = getSupplierAccountsResponseWithHeaders.getRight();
        if (responseHeaders != null && responseHeaders.containsKey("x-request-id")) {
            log.debug("[getSupplierAccounts] Response x-request-id: {}", responseHeaders.get("x-request-id").get(0));
        }
        return getSupplierAccountsResponseWithHeaders.getLeft();
    }

    public PostAssessmentsValidateAmountResponse postAssessmentsValidateAmount(String operationId, String customerKey, PostAssessmentsValidateAmountRequest postAssessmentsValidateAmountRequest) throws RestConnectorException {
        final String path = "/v1/agro-loans/assessments/customers/" + customerKey + "/validate-amount?operation-id=" + operationId;
        log.debug("[postAssessmentsValidateAmount] Request POST por validar importes del prestamo con request [{}] y path [{}]", postAssessmentsValidateAmountRequest, path);
        Pair<PostAssessmentsValidateAmountResponse, MultivaluedMap<String,Object>> postAssessmentsValidateAmountResponseWithHeaders = connector.genericPostWithResponseHeader(postAssessmentsValidateAmountRequest, PostAssessmentsValidateAmountResponse.class, CommonError.class, path, null);
        MultivaluedMap<String,Object> responseHeaders = postAssessmentsValidateAmountResponseWithHeaders.getRight();
        if (responseHeaders != null && responseHeaders.containsKey("x-request-id")) {
            log.debug("[postAssessmentsValidateAmount] Response x-request-id: {}", responseHeaders.get("x-request-id").get(0));
        }
        return postAssessmentsValidateAmountResponseWithHeaders.getLeft();
    }

    public Void postParameters(String operationId, String customerKey, Long currencyId) throws RestConnectorException {
        final String path = "/v1/agro-loans/parameters/customers/" + customerKey + "/product-currencies/" + currencyId + "?operation-id=" + operationId;
        log.debug("[postParameters] Request POST por settear parametros del prestamo con path [{}]", path);
        Pair<Void, MultivaluedMap<String,Object>> postParametersResponseWithHeaders = connector.genericPostWithResponseHeader(null, Void.class, CommonError.class, path, null);
        MultivaluedMap<String,Object> responseHeaders = postParametersResponseWithHeaders.getRight();
        if (responseHeaders != null && responseHeaders.containsKey("x-request-id")) {
            log.debug("[postParameters] Response x-request-id: {}", responseHeaders.get("x-request-id").get(0));
        }
        return postParametersResponseWithHeaders.getLeft();
    }

    public GetArrangementResponse getArrangement(String operationId, String customerKey, Long productCategoryId) throws RestConnectorException {
        final String path = "/v1/agro-loans/arrangement/customers/" + customerKey + "?operation-id=" + operationId + "&product-category-id=" + productCategoryId;
        log.debug("[getArrangementResponse] Request GET por pedir convenios disponibles con path [{}]", path);
        Pair<GetArrangementResponse, MultivaluedMap<String,Object>> getArrangementResponseWithHeaders = connector.genericGetWithResponseHeader(null, GetArrangementResponse.class, CommonError.class, path);
        MultivaluedMap<String,Object> responseHeaders = getArrangementResponseWithHeaders.getRight();
        if (responseHeaders != null && responseHeaders.containsKey("x-request-id")) {
            log.debug("[getArrangement] Response x-request-id: {}", responseHeaders.get("x-request-id").get(0));
        }
        return getArrangementResponseWithHeaders.getLeft();
    }

    public GetSimulationResponse getSimulation(String operationId, String customerKey, Long arrangementId) throws RestConnectorException {
        final String path = "/v1/agro-loans/simulation/customers-id/" + customerKey + "?operation-id=" + operationId + "&arrangment-id=" + arrangementId;
        log.debug("[getSimulationResponse] Request GET por pedir simulacion con path [{}]", path);
        Pair<GetSimulationResponse, MultivaluedMap<String,Object>> getSimulationResponseWithHeaders = connector.genericGetWithResponseHeader(null, GetSimulationResponse.class, CommonError.class, path);
        MultivaluedMap<String,Object> responseHeaders = getSimulationResponseWithHeaders.getRight();
        if (responseHeaders != null && responseHeaders.containsKey("x-request-id")) {
            log.debug("[getSimulation] Response x-request-id: {}", responseHeaders.get("x-request-id").get(0));
        }
        return getSimulationResponseWithHeaders.getLeft();
    }

    public PostApplicationResponse postApplication(String operationId, String customerKey, PostApplicationRequest postApplicationRequest) throws RestConnectorException {
        final String path = "/v1/agro-loans/applications/customers-id/" + customerKey + "?operation-id=" + operationId;
        log.debug("[postApplication] Request POST por generar prestamo con request [{}] y path [{}]", postApplicationRequest, path);
        Pair<PostApplicationResponse, MultivaluedMap<String,Object>> postApplicationResponseWithHeaders = connector.genericPostWithResponseHeader(postApplicationRequest, PostApplicationResponse.class, CommonError.class, path, null);
        MultivaluedMap<String,Object> responseHeaders = postApplicationResponseWithHeaders.getRight();
        if (responseHeaders != null && responseHeaders.containsKey("x-request-id")) {
            log.debug("[postApplications] Response x-request-id: {}", responseHeaders.get("x-request-id").get(0));
        }
        return postApplicationResponseWithHeaders.getLeft();
    }

    public GetApplicationsResponse getApplications(String operationId, String customerTaxPayerNumber, String distributorTaxPayerNumber, FilterOptions filterOptions, List<Long> productCategoriesIds, Status statusId, Date dateFrom, Date dateTo, String orderBy, String order, Integer page, Integer recordsNumber) throws RestConnectorException {
        String path = "/v1/agro-loans/applications?operation-id=" + operationId;
        if (StringUtil.isNotEmpty(customerTaxPayerNumber)) path += "&customer-taxpayer-number=" + customerTaxPayerNumber;
        if (StringUtil.isNotEmpty(distributorTaxPayerNumber)) path += "&distributor-taxpayer-number=" + distributorTaxPayerNumber;
        if (filterOptions != null) path += "&filter-options=" + filterOptions.value;
        if (CollectionUtils.isNotEmpty(productCategoriesIds)) path += "&product-categories-ids=" + productCategoriesIds.stream().map(String::valueOf).collect(Collectors.joining(","));
        if (statusId != null) path += "&status-id=" + statusId.name();
        if (dateFrom != null) path += "&date-from=" + DateUtil.formatJsonDate(dateFrom);
        if (dateTo != null) path += "&date-to=" + DateUtil.formatJsonDate(dateTo);
        if (StringUtil.isNotEmpty(orderBy)) path += "&order-by=" + orderBy;
        if (StringUtil.isNotEmpty(order)) path += "&order=" + order;
        if (page != null) path += "&page=" + page;
        if (recordsNumber != null) path += "&records-number=" + recordsNumber;

        log.debug("[getApplications] Request GET por pedir aplicaciones con path [{}]", path);
        Pair<GetApplicationsResponse, MultivaluedMap<String,Object>> getApplicationsResponseWithHeaders = connector.genericGetWithResponseHeader(null, GetApplicationsResponse.class, CommonError.class, path);
        MultivaluedMap<String,Object> responseHeaders = getApplicationsResponseWithHeaders.getRight();
        if (responseHeaders != null && responseHeaders.containsKey("x-request-id")) {
            log.debug("[getApplications] Response x-request-id: {}", responseHeaders.get("x-request-id").get(0));
        }
        return getApplicationsResponseWithHeaders.getLeft();
    }

    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        if (builder != null) {
            if (doingLogin) {
                builder.header("Authorization", RestConnectorUtil.getBasicAuthHeader(username, password));
            } else {
                builder.header("Authorization", "Bearer " + getAccessToken(false));

                // En sandbox necesitamos mandar esto para que nos devuelva info su API
                if (isSandbox) {
                    builder.header("x-request-id", "f61c2780-1f91-48f8-aed7-e687b4c617df");
                }
            }
        }
        return builder;
    }

    @Override
    public boolean retryOnUnauthorized() {
        // Solo quiero reintentar si el unauthorized me lo dio un request que no sea el de login
        if (!doingLogin) {
            try {
                getAccessToken(true);
                return true;
            } catch (RestConnectorException e) {
                log.error(e);
                return false;
            }
        }

        return false;
    }

    @Override
    public boolean getDisableHTTPSErrors() {
        return false;
    }

    public String getAccessToken(boolean force) throws RestConnectorException {
        final SimpleCache cache = SimpleCacheManager.getInstance().getCache(MacroApiConnector.class.getName());
        final String cacheKey = SimpleCacheManager.buildKey("accessToken");
        String accessToken = (String) cache.get(cacheKey);
        if (force || accessToken == null) {
            Form form = new Form();
            form.param("scope", "openid");
            form.param("grant_type", "client_credentials");

            log.debug("[getAccessToken] Por obtener token con username [{}] y password [{}]", username, password);

            doingLogin = true;
            Pair<TokenOAuth2, MultivaluedMap<String,Object>> postTokenOauth2ResponseWithHeaders = connector.genericPostWithResponseHeader(form, TokenOAuth2.class, CommonError.class, "/v1/oauth/token", MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
            MultivaluedMap<String,Object> responseHeaders = postTokenOauth2ResponseWithHeaders.getRight();
            if (responseHeaders != null && responseHeaders.containsKey("x-request-id")) {
                log.debug("[postTokenOauth2] Response x-request-id: {}", responseHeaders.get("x-request-id").get(0));
            }
            TokenOAuth2 tokenOAuth2Response = postTokenOauth2ResponseWithHeaders.getLeft();
            doingLogin = false;

            accessToken = tokenOAuth2Response.getAccessToken();
            log.debug("[getAccessToken] Token recuperado [{}]. Expira en [{}] segs. Lo agrego al a cache", accessToken, tokenOAuth2Response.getExpiresIn());
            cache.put(cacheKey, accessToken, tokenOAuth2Response.getExpiresIn() - 10);
        }

        return accessToken;
    }
}
