package ar.com.sdd.mboapi.core;

import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.WebResource;
import com.sun.jersey.api.client.config.DefaultClientConfig;
import com.sun.jersey.client.urlconnection.HttpURLConnectionFactory;
import com.sun.jersey.client.urlconnection.URLConnectionClientHandler;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.log4j.Logger;
import org.jose4j.json.internal.json_simple.JSONObject;
import org.jose4j.json.internal.json_simple.parser.JSONParser;
import org.jose4j.json.internal.json_simple.parser.ParseException;
import org.jose4j.jwe.ContentEncryptionAlgorithmIdentifiers;
import org.jose4j.jwe.JsonWebEncryption;
import org.jose4j.jwe.KeyManagementAlgorithmIdentifiers;
import org.jose4j.jws.AlgorithmIdentifiers;
import org.jose4j.jws.JsonWebSignature;
import org.jose4j.lang.JoseException;

import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLSocketFactory;
import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.MediaType;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.*;
import java.security.*;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Base64;
import java.util.Calendar;
import java.util.Date;
import java.util.Map;
import org.jose4j.jwa.AlgorithmConstraints;
import org.jose4j.jwa.AlgorithmConstraints.ConstraintType;



public class MboApiConnector implements RestSecurityManager {

    private final static Logger log = Logger.getLogger(MboApiConnector.class);
    private MboApiConnectorContext mboApiConnectorContext;
    private PrivateKey privateSignKey;
    private PublicKey  publicEncryptKey;
    private PrivateKey privateDecryptKey;
    private PublicKey  signVerifyKey;
    private boolean debugMode= false;

    private final int HttpStatus_SC_OK = 200; //Para no importar otro package solo por esto;
    private final int TokenExpirationSafeMarginSecs = 15; //Lo renuevo antes de que expire realamente
    String token;
    Date  tokenExpiration;

    public MboApiConnector(MboApiConnectorContext context) throws GeneralSecurityException, IOException {
        mboApiConnectorContext = context;
        setDebugMode(context.getDebugMode());
        if (debugMode) {
            log.debug("BaseURL:"  + context.getBaseURL());
            log.debug("Keystore:" + context.getKeystorePath());
            log.debug("ClientId:" + context.getClientId());
            log.debug("ClientAuth:" + context.getClientAuthAlias());
            log.debug("ServerAuth:" + context.getServerAuthAlias());
            log.debug("ClientCryp:" + context.getClientCrypAlias());
            log.debug("ServerCryp:" + context.getServerCrypAlias());
        }
        //Load keystores
        //1)	Load Keystore file that has all certs
            KeyStore ks = KeyStore.getInstance("JKS");
            FileInputStream fis = new FileInputStream(mboApiConnectorContext.getKeystorePath());
            ks.load(fis, mboApiConnectorContext.getKeystorePass().toCharArray());
            fis.close();

        //2)	Getting Private/Public Client Signing Key
        privateSignKey = (PrivateKey) ks.getKey(mboApiConnectorContext.getClientCrypAlias(), mboApiConnectorContext.getKeystorePass().toCharArray());
        if (debugMode) {
            X509Certificate signCert = (X509Certificate) ks.getCertificate(mboApiConnectorContext.getClientCrypAlias());
            try {
                signCert.checkValidity();
                log.debug("MBOAPI.CERT:CLIENT AUTH VALIDITY: OK");
            } catch (Exception e) {
                log.debug("MBOAPI.CERT:CLIENT AUTH VALIDITY: ERROR:" + e);
            }
        }

        //3)	Getting Public Citi Encryption Key
        X509Certificate encryptCert = (X509Certificate) ks.getCertificate(mboApiConnectorContext.getServerCrypAlias());
        if (debugMode) {
            try {
                encryptCert.checkValidity();
                log.debug("MBOAPI.CERT:SERVER CRYPT VALIDITY: OK");
            } catch (Exception e) {
                log.debug("MBOAPI.CERT:SERVER CRYPT VALIDITY: ERROR:" + e);
            }
        }
        publicEncryptKey = encryptCert.getPublicKey();

        //4)  Getting Private Client DecryptKeys
        privateDecryptKey = (PrivateKey) ks.getKey(mboApiConnectorContext.getClientCrypAlias(), mboApiConnectorContext.getKeystorePass().toCharArray());
        if (debugMode) {
            try {
                X509Certificate decryptCert = (X509Certificate)ks.getCertificate(mboApiConnectorContext.getClientCrypAlias());
                decryptCert.checkValidity();
                log.debug("MBOAPI.CERT:CLIENT CRYPT VALIDITY: OK");
            } catch (Exception e) {
                log.debug("MBOAPI.CERT:CLIENT CRYPT VALIDITY: ERROR:" + e);
            }
        }

        //e)	Getting Public Citi Verification Key
        X509Certificate signVerifyCert = (X509Certificate) ks.getCertificate(mboApiConnectorContext.getServerAuthAlias());
        if (debugMode) {
            try {
                signVerifyCert.checkValidity();
                log.debug("MBOAPI.CERT:SERVER SIGN VALIDITY: OK");
            } catch (Exception e) {
                log.debug("MBOAPI.CERT:SERVER SIGN VALIDITY: ERROR:" + e);
            }
        }
        signVerifyKey = signVerifyCert.getPublicKey();

        SSLSocketFactory socketFactory = new ar.com.sdd.commons.rest.util.SSLSocketFactoryGenerator(mboApiConnectorContext.getClientAuthAlias(),mboApiConnectorContext.getKeystorePath(), mboApiConnectorContext.getKeystorePass()).getSSLSocketFactory();
        HttpsURLConnection.setDefaultSSLSocketFactory(socketFactory);


        tokenExpiration = new Date(2001-1900, Calendar.JANUARY, 1); //En el pasado

    }

    public String encrypt (String payload) throws JoseException {
        if (debugMode) log.debug("MBOAPI.PAYLOAD PLAIN:"+payload);

        JsonWebSignature jwsignature = new JsonWebSignature();
        jwsignature.setPayload(payload);
        jwsignature.setAlgorithmHeaderValue(AlgorithmIdentifiers.RSA_USING_SHA256);
        jwsignature.setKey(privateSignKey);

        String oAuthPayloadSigned = jwsignature.getCompactSerialization();
        //System.out.println(oAuthPayloadSigned);

        //4)	Encrypt the Signed Payload
        JsonWebEncryption jwEncrypt = new JsonWebEncryption();
        jwEncrypt.setPlaintext(oAuthPayloadSigned);
        jwEncrypt.setAlgorithmHeaderValue(KeyManagementAlgorithmIdentifiers.RSA_OAEP_256);
        jwEncrypt.setEncryptionMethodHeaderParameter(ContentEncryptionAlgorithmIdentifiers.AES_128_CBC_HMAC_SHA_256);
        jwEncrypt.setKey(publicEncryptKey);
        String payloadSignedEncrypted = jwEncrypt.getCompactSerialization();
        if (debugMode) log.debug("MBOAPI.PAYLOAD SIGN+CRYPT:"+payloadSignedEncrypted);
        return payloadSignedEncrypted;
    }

    public String decrypt(String responsePayload) throws JoseException {
        // d)	Decrypt the encrypted & signed Response Payload
        JsonWebEncryption jwEncryption = new JsonWebEncryption();
        if (debugMode) {
            log.debug("MBOAPI.PAYLOAD ENCRYPTED KEY:" + privateDecryptKey);
            log.debug("MBOAPI.PAYLOAD ENCRYPTED RESPONSE:" + responsePayload);
        }
        jwEncryption.setKey(privateDecryptKey);
        jwEncryption.setCompactSerialization(responsePayload);
        jwEncryption.setAlgorithmConstraints(new AlgorithmConstraints(ConstraintType.PERMIT, KeyManagementAlgorithmIdentifiers.RSA1_5
                                                                                            ,KeyManagementAlgorithmIdentifiers.RSA_OAEP_256)); // Fix!

        String decryptedResponse = jwEncryption.getPlaintextString();
        if (debugMode) {
            log.debug("MBOAPI.PAYLOAD DECRYPTED RESPONSE:" + decryptedResponse);
        }

        //e)	Verifying the Signature of decrypted Response
        JsonWebSignature jwSignature = new JsonWebSignature();
        jwSignature.setKey(signVerifyKey);
        jwSignature.setCompactSerialization(decryptedResponse);
        if (debugMode) {
            System.setProperty("org.jose4j.jws.getPayload-skip-verify","true");
        }
        String verifiedResponse = jwSignature.getPayload();
        if (debugMode) {
            log.debug("MBOAPI.PAYLOAD VERIFIED RESPONSE:" + verifiedResponse);
        }
        return verifiedResponse;
    }

    private Client createClient() throws GeneralSecurityException, IOException {
        //Auth API
        SSLSocketFactory socketFactory = new ar.com.sdd.commons.rest.util.SSLSocketFactoryGenerator(mboApiConnectorContext.getClientAuthAlias(),mboApiConnectorContext.getKeystorePath(), mboApiConnectorContext.getKeystorePass()).getSSLSocketFactory();
        HttpsURLConnection.setDefaultSSLSocketFactory(socketFactory);
        String proxyURL="";
        Client client = new Client(new URLConnectionClientHandler(
                new HttpURLConnectionFactory() {
                    Proxy proxy = null;
                    public HttpURLConnection getHttpURLConnection(URL url)
                            throws IOException {
                        if(proxy==null && !proxyURL.isEmpty()){
                            proxy = new Proxy(Proxy.Type.HTTP, new InetSocketAddress(proxyURL, 8080));
                        }else{
                            proxy=Proxy.NO_PROXY;
                        }
                        return (HttpURLConnection) url.openConnection(proxy);
                    }
                }), new DefaultClientConfig());
        return client;
    }

    private  String postOauthPayload(String apiURL, String payload) throws JoseException, GeneralSecurityException, IOException {
        String payloadSignedEncrypted = encrypt(payload);
        Client client = createClient();

        String clientID= mboApiConnectorContext.getClientId();
        String clientSecret= mboApiConnectorContext.getClientSecret();
        WebResource webResource = client.resource(apiURL);
        WebResource.Builder  builder = webResource.type(MediaType.APPLICATION_JSON_TYPE).accept(MediaType.APPLICATION_JSON);
        String authHeader = "Basic " + Base64.getEncoder().encodeToString(((clientID+":"+clientSecret).getBytes())).replaceAll("[\\r|\\n]", "");
        builder.header(HttpHeaders.AUTHORIZATION, authHeader);
        ClientResponse clientResponse = builder.post(ClientResponse.class, payloadSignedEncrypted);

        RestConnectorEnvironment environment = new RestConnectorEnvironment();
        RestConnector restConnector = new RestConnector(environment, this);

        if (debugMode) {
            log.debug("MBO.REQUEST HEADER:" + HttpHeaders.AUTHORIZATION + " "+ authHeader);
        }
        if (clientResponse.getStatus() != HttpStatus_SC_OK) {
            log.error("MBO.REQUEST RESPONSE HEADER:" + clientResponse.getHeaders());
        }
        log.debug("MBO.REQUEST APIM-GUID:"+clientResponse.getHeaders().get("apim-guid"));
        String responsePayload = clientResponse.getEntity(String.class);
        String resultPayload =  decrypt(responsePayload);
        if (debugMode) {
            log.debug("MBO.PAYLOAD RESULT:" + resultPayload);
        }
        return resultPayload;
    }

    private  String postPayload(String apiURL, String payload) throws JoseException, GeneralSecurityException, IOException {
        String payloadSignedEncrypted = encrypt(payload);
        Client client = createClient();

        String clientID= mboApiConnectorContext.getClientId();
        WebResource webResource = client.resource(apiURL).queryParam("client_id", clientID);
        WebResource.Builder  builder = webResource.type(MediaType.APPLICATION_JSON_TYPE).accept(MediaType.APPLICATION_JSON);
        String authHeader = "Bearer " + getToken(false);
        builder.header(HttpHeaders.AUTHORIZATION,authHeader);
        if (debugMode) {
            log.debug("MBO.REQUEST HEADER:" + HttpHeaders.AUTHORIZATION + " "+ authHeader);
        }
        ClientResponse clientResponse = builder.post(ClientResponse.class, payloadSignedEncrypted);

        if (clientResponse.getStatus() != HttpStatus_SC_OK) {
            log.error("MBO.REQUEST RESPONSE HEADER:" + clientResponse.getHeaders());
        }
        log.debug("MBO.REQUEST APIM-GUID:"+clientResponse.getHeaders().get("apim-guid"));
        String responsePayload = clientResponse.getEntity(String.class);
        String resultPayload =  decrypt(responsePayload);
        if (debugMode) {
            log.debug("MBO.PAYLOAD RESULT:" + resultPayload);
        }
        return resultPayload;
    }

    public String getToken(boolean force) throws JoseException, GeneralSecurityException, IOException {
        final String apiPath="/openbankingservices/v1/oauth/token";
       /* Espera:
            {"oAuthToken":
                {"grantType": "client_credentials",
                 "scope": "/openbanking/v1"
                 }}
       */
        JSONObject oAuthToken = new JSONObject();
        oAuthToken.put("grantType", "client_credentials");
        oAuthToken.put("scope"    , "/openbanking/v1");
        JSONObject payload = new JSONObject();
        payload.put("oAuthToken", oAuthToken);

        if (!force && (new Date()).before( tokenExpiration)) return token;

        //Busco un nuevo token:
        String result =  postOauthPayload(normalizeURL(mboApiConnectorContext.getBaseURL(), apiPath),payload.toJSONString());
        /* Un token correcto es asi:
            { "token":
               { "token_type":"Bearer",
                   "access_token":"FJEdKNJeQsDmUAtgKcefWPdD6SbVUvXiL4hjShm+hU1l9Q2e0QqKfTg8lsW2QgS3rN7X67JkkrrNK/xXql8GzuB/ev4+ObQo7fGHn71QZYYgWFZ2JByMyHcyVXjJFP/eMV0v2tOr5K2vslmJSGV/k7+Kdv7G2Flu0Nt5mGHP+ltTdDhGW5xAkTslH7QuJOZhTCksZQmm/WyoO8Qx7XlaQeRqZGXTkY2KJoY/LaQ1mhUJEkqGaoV80ze6WtTI4G70eafrcafVm0xDgyHkrK3PbpByvc217eysnwYeMwCe6Y0=",
                   "expires_in":"1800",
                   "scope":"/openbanking/v1" } }
         */
        try {
            JSONParser jsonParser = new JSONParser();
            JSONObject oToken = (JSONObject)jsonParser.parse(result);
            Map<String,String> oTokenToken = (Map<String,String>)oToken.get("token");
            token = oTokenToken.get("access_token");
            tokenExpiration = DateUtils.addSeconds(new Date(), Integer.parseInt(oTokenToken.get("expires_in"))- TokenExpirationSafeMarginSecs);
        } catch (Exception e) {
            log.error("Obteniendo el token: "+ e);
            log.debug("PAYLOAD:"+ result);
        }

        return token;
    }


    public JSONObject postStatusReport(String payload) throws JoseException, ParseException, GeneralSecurityException, IOException {
        final String apiPath = "/openbanking/v1/requesttopay/status";

        String result =  postPayload(normalizeURL(mboApiConnectorContext.getBaseURL(),apiPath),payload);
        JSONParser jsonParser = new JSONParser();
        return (JSONObject)jsonParser.parse(result);
    }

    public boolean isDebugMode() {
        return debugMode;
    }

    public void setDebugMode(boolean debugMode) {
        this.debugMode = debugMode;
    }

    //Si uso por error doble barra, no anda nada
    public static String normalizeURL(String baseUrl, String apiUrl) {
        try {
            URI  uri = new URI(baseUrl+"/"+apiUrl);
            return uri.normalize().toString();
        } catch (Exception e) {
            log.error(e);
        }
        return baseUrl+apiUrl;
    }

    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        return  builder;
    }

    @Override
    public boolean retryOnUnauthorized() {
        //No estoy usando los metodos de post de restoConnector
        return false;
    }

    @Override
    public boolean getDisableHTTPSErrors() {
        return false;
    }


}