package ar.com.sdd.mercadopago.core;

import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import ar.com.sdd.commons.rest.model.TokenOAuth2;
import ar.com.sdd.commons.rest.util.QueryBuilder;
import ar.com.sdd.mercadopago.entity.*;
import ar.com.sdd.mercadopago.entity.user.User;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;

import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.Form;
import javax.ws.rs.core.MediaType;

public class MercadoPagoConnector implements RestSecurityManager {

    private static final Logger log = Logger.getLogger(MercadoPagoConnector.class);

    private static final String MP_BASE_URL = "https://api.mercadopago.com";

    public Preference savePreference(String token, Preference request) throws RestConnectorException {
        log.debug("[savePreference] Request POST preferencia [" + request + "]");
        Preference response = getNewRestConnector().genericPost(request, Preference.class, MPError.class, "/checkout/preferences?access_token=" + token);
        log.debug("[savePreference] Response POST de la preferencia [" + response + "]");
        return response;
    }

    public MPPaging<Payment> searchPayment(String token, String externalReference) throws RestConnectorException {
        log.debug("[searchPayment] Request con externalReference [" + externalReference + "]");
        MPPaging<Payment> response = getNewRestConnector().genericGet(null, MPPaging.MPPagingPayment.class, MPError.class, "/v1/payments/search?external_reference=" + externalReference + "&access_token=" + token);
        log.debug("[searchPayment] Response con externalReference [" + externalReference + "],  [" + response + "]");
        return response;
    }

    public TokenOAuth2 getToken(String clientId, String clientSecret) throws RestConnectorException {
        Form form = new Form();
        form.param("client_id", clientId);
        form.param("client_secret", clientSecret);
        form.param("grant_type", "client_credentials");

        log.debug("[getToken] Por recuperar token con clientId [" + clientId + "], clientSecret [" + clientSecret + "]");

        RestConnectorEnvironment environment = new RestConnectorEnvironment();
        environment.baseUrl = MP_BASE_URL;
        RestConnector connector = new RestConnector(environment, null);
        TokenOAuth2 tokenOAuth2Response = connector.genericPost(form, TokenOAuth2.class, "/oauth/token", MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);

        log.debug("[getToken] Token recuperado [" + tokenOAuth2Response + "]");
        return tokenOAuth2Response;
    }

    public MPTokenOAuth2 getAuthorizationCodeToken(String clientId, String clientSecret, String mercadoPagoId, String redirectUri, String refreshToken) throws RestConnectorException {
        Form form = new Form();
        if (StringUtils.isNotEmpty(clientId)) {
            form.param("client_id", clientId);
        }
        form.param("client_secret", clientSecret);
        form.param("code", mercadoPagoId);
        form.param("redirect_uri", redirectUri);
        form.param("grant_type", "authorization_code");

        //Si vino el refresh token, cambio el grant_type para la operacion de refresh
        if (StringUtils.isNotEmpty(refreshToken)) {
            form.param("refresh_token", refreshToken);
            form.param("grant_type", "authorization_code");
        }

        log.debug("[getAuthorizationCodeToken] Por recuperar token con clientId [" + clientId + "], clientSecret [" + clientSecret + "], mercadoPagoId [" + mercadoPagoId + "], redirectUri [" + redirectUri + "], refreshToken [" + refreshToken +"]");
        MPTokenOAuth2 tokenOAuth2Response = getNewRestConnectorNoSecurityManager().genericPost(form, MPTokenOAuth2.class, "/oauth/token", MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
        log.debug("[getAuthorizationCodeToken] Token recuperado [" + tokenOAuth2Response + "]");
        return tokenOAuth2Response;

    }

    public User getUsersMe(String userAccessToken) throws RestConnectorException {
        log.debug("[getUsersMe] Por recuperar usuario con userAccessToken [" + userAccessToken + "]");
        QueryBuilder queryBuilder = QueryBuilder.Builder().path("/users/me").add("access_token", userAccessToken);
        User user = getNewRestConnectorNoSecurityManager().genericGet(Void.class, User.class, queryBuilder.build());
        log.debug("[getUsersMe] User recuperado [" + user + "]");
        return user;
    }

    public Payment postPayments(Payment payment, String mpTokenOAuthAccessToken) throws RestConnectorException {
        log.debug("[postPayments] Por enviar el pago [" + payment + "] para el accessToken OAuth2 [" + mpTokenOAuthAccessToken + "]");
        QueryBuilder queryBuilder = QueryBuilder.Builder().path("/v1/payments").add("access_token", mpTokenOAuthAccessToken);
        Payment paymentResponse = getNewRestConnectorNoSecurityManager().genericPost(payment, Payment.class, queryBuilder.build());
        log.debug("[postPayments] Payment devuelto [" + paymentResponse + "]");
        return paymentResponse;
    }



    private RestConnector getNewRestConnector() {
        RestConnectorEnvironment environment = new RestConnectorEnvironment();
        environment.baseUrl = MP_BASE_URL;
        return new RestConnector(environment,this);
    }

    private RestConnector getNewRestConnectorNoSecurityManager() {
        RestConnectorEnvironment environment = new RestConnectorEnvironment();
        environment.baseUrl = MP_BASE_URL;
        return new RestConnector(environment,null);
    }

    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        //Devuelvo el mismo builder sin modificarlo
        return builder;
    }


}
