package ar.com.sdd.mercadopago.entity;

import ar.com.sdd.mercadopago.entity.preference.BackUrls;
import ar.com.sdd.mercadopago.entity.preference.DifferentialPricing;
import ar.com.sdd.mercadopago.entity.preference.Item;
import ar.com.sdd.mercadopago.entity.preference.Payer;
import ar.com.sdd.mercadopago.entity.preference.PaymentMethods;
import ar.com.sdd.mercadopago.entity.preference.Shipments;
import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.annotation.JsonNaming;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Objects;

/**
 * Mercado Pago MercadoPago
 * This resource allows you to set up, during the Payment process, all the item information,
 * any accepted means of Payment and many other options.
 *
 * Los paymentMethos se pueden restringir excluyento paymenMethod individuales, o por ctegorias de paymenrMethodType
 * La version actualizada esta en :
 *                  https://www.mercadopago.com.ar/developers/es/guides/resources/localization/payment-methods
 *
 *
 * Medio de pago	           paymentType ID    paymenMethod ID
 * Visa	                        credit_card	    visa
 * Mastercard	                credit_card	    master
 * American Express	            credit_card	    amex
 * Diners Club International	credit_card	    diners
 * Tarjeta Naranja	            credit_card	    naranja
 * Tarjeta Nativa	            credit_card	    nativa
 * Tarjeta Shopping	            credit_card	    shopping
 * Tarjeta Cencosud	            credit_card	    cencosud
 * Tarjeta CMR MasterCard	    credit_card	    cmr_master
 * Argencard	                credit_card	    argencard
 * Cordial	                    credit_card	    cordial
 * Cordobesa	                credit_card	    cordobesa
 * Cabal	                    credit_card	    cabal
 * Visa Débito	                debit_card	    debvisa
 * Mastercard Débito	        debit_card	    debmaster
 * Maestro	                    debit_card	    maestro
 * Cabal Débito	                debit_card	    debcabal
 * Pago Fácil	                ticket	        pagofacil
 * RapiPago	                    ticket	        rapipago
 * Carga Virtual	            ticket	        cargavirtual
 * Cobro Express	            ticket	        cobroexpress
 * Red Link	                    atm	            redlink
 * Dinero en cuenta	            account_money	account_money
 *
 *
 * <p>
 * Created by Eduardo Paoletta on 11/9/16.
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
@JsonNaming(PropertyNamingStrategy.SnakeCaseStrategy.class)
public class Preference {

    public enum OperationType {
        regular_payment,
        money_transfer
    }

    public enum AutoReturn {
        approved,
        all
    }

    private ArrayList<Item> items = null;
    private Payer payer = null;
    private PaymentMethods paymentMethods = null;
    private Shipments shipments = null;

    private  String statementDescriptor = null;

    private BackUrls backUrls = null;
    private String notificationUrl = null;
    private String id = null;
    private String initPoint = null;
    private String sandboxInitPoint = null;
    private Date dateCreated = null;
    private OperationType operationType = null;
    private String additionalInfo = null;
    private AutoReturn autoReturn = null;
    private String externalReference = null;
    private Boolean expires = null;
    @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSSXXX", timezone = "GMT-03:00")
    private Date expirationDateFrom = null;
    @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSSXXX", timezone = "GMT-03:00")
    private Date expirationDateTo = null;
    @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSSXXX", timezone = "GMT-03:00")
    private Date dateOfExpiration = null; // Se usa solo para el PUT de preference
    private BigDecimal collectorId = null;
    private BigDecimal clientId = null;
    private String marketplace = null;
    private BigDecimal marketplaceFee = null;
    private DifferentialPricing differentialPricing = null;


    public ArrayList<Item> getItems() {
        return items;
    }

    public Preference setItems(ArrayList<Item> items) {
        this.items = items;
        return this;
    }

    public Preference appendItem(Item item) {
        if (items == null) {
            items = new ArrayList<Item>();
        }
        items.add(item);
        return this;
    }

    public Payer getPayer() {
        return payer;
    }

    /**
     * Información del comprador, como nombre, apellido, correo electrónico, teléfono, identificación personal, dirección y fecha de registro.
     */
    public Preference setPayer(Payer payer) {
        this.payer = payer;
        return this;
    }

    public PaymentMethods getPaymentMethods() {
        return paymentMethods;
    }

    /**
     * Toda la configuración relacionada con los métodos de pago, tales como los métodos de pago excluidos,
     * los tipos de pago excluidos, el método de pago predeterminado y las cuotas.
     *
     * excluded_payment_methods  : Métodos de pago excluidos del proceso de pago (excepto account_money, que siempre estará disponible). Los métodos de pago incluidos aquí no estarán disponibles en el checkout.
     * excluded_payment_types    : Tipos de pago excluidos del proceso de pago. Los tipos de pago incluidos aquí no estarán disponibles en el checkout.
     * default_payment_method_id : Forma de pago sugerida. El usuario iniciará el pago con este método de pago ya seleccionado. Se considera sugerida porque los Checkouts siguen lógicas diferentes para seleccionar el mejor método de pago para una transacción específica, siendo esta propiedad la opción con menor relevancia.
     * installments              : Máximo número de cuotas
     * default_installments      : Preferencia de cuotas
     *
     */
    public Preference setPaymentMethods(PaymentMethods paymentMethods) {
        this.paymentMethods = paymentMethods;
        return this;
    }

    public Shipments getShipments() {
        return shipments;
    }

    public Preference setShipments(Shipments shipments) {
        this.shipments = shipments;
        return this;
    }

    /**
     * El statement descriptor es un texto largo (hasta 16 caracteres) que se visualizará en la factura de la tarjeta de crédito del pagador para identificar fácilmente la compra.
     */
    public void setStatementDescriptor(String statementDescriptor) {
        this.statementDescriptor = statementDescriptor;
    }

    public String getStatementDescriptor() {
        return statementDescriptor;
    }

    public BackUrls getBackUrls() {
        return backUrls;
    }

    public Preference setBackUrls(BackUrls backUrls) {
        this.backUrls = backUrls;
        return this;
    }

    public String getNotificationUrl() {
        return notificationUrl;
    }

    /**
     * URL a la cual te gustaría recibir notificaciones de pagos
     */
    public Preference setNotificationUrl(String notificationUrl) {
        this.notificationUrl = notificationUrl;
        return this;
    }

    public String getId() {
        return id;
    }

    public void setId(String id) {
        this.id = id;
    }

    public String getInitPoint() {
        return initPoint;
    }

    public void setInitPoint(String initPoint) {
        this.initPoint = initPoint;
    }

    public String getSandboxInitPoint() {
        return sandboxInitPoint;
    }

    public void setSandboxInitPoint(String sandboxInitPoint) {
        this.sandboxInitPoint = sandboxInitPoint;
    }

    public Date getDateCreated() {
        return dateCreated;
    }

    public void setDateCreated(Date dateCreated) {
        this.dateCreated = dateCreated;
    }

    public OperationType getOperationType() {
        return operationType;
    }

    public void setOperationType(OperationType operationType) {
        this.operationType = operationType;
    }

    public String getAdditionalInfo() {
        return additionalInfo;
    }

    public Preference setAdditionalInfo(String additionalInfo) {
        this.additionalInfo = additionalInfo;
        return this;
    }

    public AutoReturn getAutoReturn() {
        return autoReturn;
    }

    /**
     *En el caso de estar especificado tu comprador sera redirigido a tu sitio inmediatamente después de la compra
     * approved: The redirection takes place only for approved payments.
     * all: The redirection takes place only for approved payments, forward compatibility only if we change the default behavior
     */
    public Preference setAutoReturn(AutoReturn autoReturn) {
        this.autoReturn = autoReturn;
        return this;
    }

    public String getExternalReference() {
        return externalReference;
    }

    public Preference setExternalReference(String externalReference) {
        this.externalReference = externalReference;
        return this;
    }

    public Boolean getExpires() {
        return expires;
    }

    /**
     * Preferencia que determina si una preferencia expira
     */
    public Preference setExpires(Boolean expires) {
        this.expires = expires;
        return this;
    }


    public Date getExpirationDateFrom() {
        return expirationDateFrom;
    }

    /**
     * Fecha en formato "yyyy-MM-dd'T'HH:mm:ssz" que indica el inicio del período de validez de la preferencia.
     * Esto se puede usar, por ejemplo, para ventas limitadas, donde los vendedores hacen una oferta entre ciertas fechas.
     * Por ejemplo - 2022-11-17T09:37:52.000-04:00.
     */
    public Preference setExpirationDateFrom(Date expirationDateFrom) {
        this.expirationDateFrom = expirationDateFrom;
        return this;
    }

    public Date getExpirationDateTo() {
        return expirationDateTo;
    }

    /**
     * Fecha en formato "yyyy-MM-dd'T'HH:mm:ssz" que indica el final del período de validez de la preferencia.
     * Esto se puede usar, por ejemplo, para ventas limitadas, donde los vendedores hacen una oferta entre ciertas fechas.
     * Por ejemplo - 2022-11-17T09:37:52.000-04:00.
     */
    public Preference setExpirationDateTo(Date expirationDateTo) {
        this.expirationDateTo = expirationDateTo;
        return this;
    }

    /**
     * Fecha de vencimiento del medio de pago en efectivo
     */
    public Date getDateOfExpiration() {
        return dateOfExpiration;
    }

    public Preference setDateOfExpiration(Date dateOfExpiration) {
        this.dateOfExpiration = dateOfExpiration;
        return this;
    }

    public BigDecimal getCollectorId() {
        return collectorId;
    }

    public void setCollectorId(BigDecimal collectorId) {
        this.collectorId = collectorId;
    }

    public BigDecimal getClientId() {
        return clientId;
    }

    public void setClientId(BigDecimal clientId) {
        this.clientId = clientId;
    }

    public String getMarketplace() {
        return marketplace;
    }

    /**
     * Origen del pago.
     * Este es un campo alfanumérico cuyo valor por defecto es NONE.
     * Si el collector tiene su propio marketplace, aquí es donde se envían las credenciales para identificarlo.
     * Como el marketplace está asociado al Application ID, las credenciales del marketplace deben corresponder a las credenciales utilizadas
     * para crear la preferencia. El uso de credenciales incorrectas dará como resultado un error.
     *
     */
    public Preference setMarketplace(String marketplace) {
        this.marketplace = marketplace;
        return this;
    }

    public BigDecimal getMarketplaceFee() {
        return marketplaceFee;
    }

    /**
     * Tarifa de Marketplace cobrada por el propietario de la aplicación. Es una cantidad fija y su valor por defecto es 0 en moneda local.
     * Esta propiedad solo se puede enviar si también se ha definido un marketplace válido; de lo contrario, la solicitud fallará.
     */
    public Preference setMarketplaceFee(BigDecimal marketplaceFee) {
        this.marketplaceFee = marketplaceFee;
        return this;
    }

    public DifferentialPricing getDifferentialPricing() {
        return differentialPricing;
    }

    /**
     * Differential pricing configuration for this preference.
     * The seller can have their own configuration regarding payment methods and installments,
     * and it`s in this parameter where they will include the numeric ID to indicate so.
     * This value is used to apply custom payment methods.
     */
    public Preference setDifferentialPricing(DifferentialPricing differentialPricing) {
        this.differentialPricing = differentialPricing;
        return this;
    }

    @Override
    public String toString() {
        return "Preference [" +
                "items=" + (items == null ? items : Arrays.toString(items.toArray())) +
                ", payer=" + payer +
                ", paymentMethods=" + paymentMethods +
                ", shipments=" + shipments +
                ", statementDescriptor=" + statementDescriptor +
                ", backUrls=" + backUrls +
                ", notificationUrl=" + notificationUrl +
                ", id=" + id +
                ", initPoint=" + initPoint +
                ", sandboxInitPoint=" + sandboxInitPoint +
                ", dateCreated=" + dateCreated +
                ", operationType=" + operationType +
                ", additionalInfo=" + additionalInfo +
                ", autoReturn=" + autoReturn +
                ", externalReference=" + externalReference +
                ", expires=" + expires +
                ", expirationDateFrom=" + expirationDateFrom +
                ", expirationDateTo=" + expirationDateTo +
                ", dateOfExpiration=" + dateOfExpiration +
                ", collectorId=" + collectorId +
                ", clientId=" + clientId +
                ", marketplace=" + marketplace +
                ", marketplaceFee=" + marketplaceFee +
                ", differentialPricing=" + differentialPricing +
                ']';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        Preference that = (Preference) o;

        if (!Objects.equals(items, that.items)) return false;
        if (!Objects.equals(payer, that.payer)) return false;
        if (!Objects.equals(paymentMethods, that.paymentMethods))
            return false;
        if (!Objects.equals(shipments, that.shipments)) return false;
        if (!Objects.equals(statementDescriptor, that.statementDescriptor))
            return false;
        if (!Objects.equals(backUrls, that.backUrls)) return false;
        if (!Objects.equals(notificationUrl, that.notificationUrl))
            return false;
        if (!Objects.equals(id, that.id)) return false;
        if (!Objects.equals(initPoint, that.initPoint)) return false;
        if (!Objects.equals(sandboxInitPoint, that.sandboxInitPoint))
            return false;
        if (!Objects.equals(dateCreated, that.dateCreated)) return false;
        if (operationType != that.operationType) return false;
        if (!Objects.equals(additionalInfo, that.additionalInfo))
            return false;
        if (autoReturn != that.autoReturn) return false;
        if (!Objects.equals(externalReference, that.externalReference))
            return false;
        if (!Objects.equals(expires, that.expires)) return false;
        if (!Objects.equals(expirationDateFrom, that.expirationDateFrom))
            return false;
        if (!Objects.equals(expirationDateTo, that.expirationDateTo))
            return false;
        if (!Objects.equals(dateOfExpiration, that.dateOfExpiration))
            return false;
        if (!Objects.equals(collectorId, that.collectorId)) return false;
        if (!Objects.equals(clientId, that.clientId)) return false;
        if (!Objects.equals(marketplace, that.marketplace)) return false;
        if (!Objects.equals(marketplaceFee, that.marketplaceFee))
            return false;
        return Objects.equals(differentialPricing, that.differentialPricing);
    }

    @Override
    public int hashCode() {
        int result = items != null ? items.hashCode() : 0;
        result = 31 * result + (payer != null ? payer.hashCode() : 0);
        result = 31 * result + (paymentMethods != null ? paymentMethods.hashCode() : 0);
        result = 31 * result + (shipments != null ? shipments.hashCode() : 0);
        result = 31 * result + (statementDescriptor != null ? statementDescriptor.hashCode() : 0);
        result = 31 * result + (backUrls != null ? backUrls.hashCode() : 0);
        result = 31 * result + (notificationUrl != null ? notificationUrl.hashCode() : 0);
        result = 31 * result + (id != null ? id.hashCode() : 0);
        result = 31 * result + (initPoint != null ? initPoint.hashCode() : 0);
        result = 31 * result + (sandboxInitPoint != null ? sandboxInitPoint.hashCode() : 0);
        result = 31 * result + (dateCreated != null ? dateCreated.hashCode() : 0);
        result = 31 * result + (operationType != null ? operationType.hashCode() : 0);
        result = 31 * result + (additionalInfo != null ? additionalInfo.hashCode() : 0);
        result = 31 * result + (autoReturn != null ? autoReturn.hashCode() : 0);
        result = 31 * result + (externalReference != null ? externalReference.hashCode() : 0);
        result = 31 * result + (expires != null ? expires.hashCode() : 0);
        result = 31 * result + (expirationDateFrom != null ? expirationDateFrom.hashCode() : 0);
        result = 31 * result + (expirationDateTo != null ? expirationDateTo.hashCode() : 0);
        result = 31 * result + (dateOfExpiration != null ? dateOfExpiration.hashCode() : 0);
        result = 31 * result + (collectorId != null ? collectorId.hashCode() : 0);
        result = 31 * result + (clientId != null ? clientId.hashCode() : 0);
        result = 31 * result + (marketplace != null ? marketplace.hashCode() : 0);
        result = 31 * result + (marketplaceFee != null ? marketplaceFee.hashCode() : 0);
        result = 31 * result + (differentialPricing != null ? differentialPricing.hashCode() : 0);
        return result;
    }
}
