package ar.com.sdd.mercadopago.core;

import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import ar.com.sdd.commons.rest.model.TokenOAuth2;
import ar.com.sdd.commons.rest.util.QueryBuilder;
import ar.com.sdd.mercadopago.entity.*;
import ar.com.sdd.mercadopago.entity.account.MercadoPagoReportConfiguration;
import ar.com.sdd.mercadopago.entity.account.MercadoPagoReportResponse;
import ar.com.sdd.mercadopago.entity.account.MercadoPagotReportRequest;
import ar.com.sdd.mercadopago.entity.user.User;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.Form;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.core.Request;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

public class MercadoPagoConnector implements RestSecurityManager {

    private static final Logger log = LogManager.getLogger(MercadoPagoConnector.class);

    private String mpBaseUrl;

    private String accessToken;

    public MercadoPagoConnector(String accessToken) {
        this(null, accessToken);
    }
    public MercadoPagoConnector(String MPBaseUrl, String accessToken) {
        this.mpBaseUrl =MPBaseUrl!=null?MPBaseUrl:"https://api.mercadopago.com";
        this.accessToken = accessToken;
    }

    public Preference savePreference(Preference request) throws RestConnectorException {
        log.debug("[savePreference] Request POST preferencia [" + request + "]");
        Preference response = getConnector().genericPost(request, Preference.class, MPError.class, "/checkout/preferences?access_token=" + accessToken);
        log.debug("[savePreference] Response POST de la preferencia [" + response + "]");
        return response;
    }

    public Preference updatePreference(Preference request) throws RestConnectorException {
        log.debug("[updatePreference] Request PUT preferencia [" + request + "]");
        String id = request.getId();
        Preference response = getConnector().genericPut(request, Preference.class, MPError.class, "/checkout/preferences/" + id + "?access_token=" + accessToken);
        log.debug("[updatePreference] Response PUT de la preferencia [" + response + "]");
        return response;
    }

    public MPPaging<Payment> searchPayment(String externalReference) throws RestConnectorException {
        log.debug("[searchPayment] Request con externalReference [" + externalReference + "]");
        MPPaging<Payment> response = getConnector().genericGet(null, MPPaging.MPPagingPayment.class, MPError.class, "/v1/payments/search?external_reference=" + externalReference + "&access_token=" + accessToken);
        log.debug("[searchPayment] Response con externalReference [" + externalReference + "],  [" + response + "]");
        return response;
    }

    public MPPaging<Payment> searchPaymentById(String id) throws RestConnectorException {
        log.debug("[searchPayment] Request con id [" + id + "]");
        MPPaging<Payment> response = getConnector().genericGet(null, MPPaging.MPPagingPayment.class, MPError.class, "/v1/payments/" + id + "&access_token=" + accessToken);
        log.debug("[searchPayment] Response con id [" + id + "],  [" + response + "]");
        return response;
    }


    public MPPaging<MerchantOrder> searchMerchantOrderById(String id) throws RestConnectorException {
        log.debug("[searchMerchantOrder] Request con id [" + id + "]");
        MPPaging<MerchantOrder> response = getConnector().genericGet(null, MPPaging.MPPagingMerchantOrder.class, MPError.class, "/merchant_orders/" + id + "&access_token=" + accessToken);
        log.debug("[searchMerchantOrder] Response con id [" + id + "],  [" + response + "]");
        return response;
    }

    public TokenOAuth2 getToken(String clientId, String clientSecret) throws RestConnectorException {
        Form form = new Form();
        form.param("client_id", clientId);
        form.param("client_secret", clientSecret);
        form.param("grant_type", "client_credentials");

        log.debug("[getToken] Por recuperar token con clientId [" + clientId + "], clientSecret [" + clientSecret + "]");

        RestConnectorEnvironment environment = new RestConnectorEnvironment();
        environment.baseUrl = mpBaseUrl;
        RestConnector connector = new RestConnector(environment, null);
        TokenOAuth2 tokenOAuth2Response = connector.genericPost(form, TokenOAuth2.class, "/oauth/token", MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);

        log.debug("[getToken] Token recuperado [" + tokenOAuth2Response + "]");
        return tokenOAuth2Response;
    }

    /*
        No parece ser usado en ningun lado
     */
    public MPTokenOAuth2 getAuthorizationCodeToken(String environment, String clientId, String clientSecret, String applicationAccessToken, String mercadoPagoId, String redirectUri, String refreshToken) throws RestConnectorException {
        log.debug("[getAuthorizationCodeToken] Entrando en getAuthorizationCodeToken con environment [" + environment + "], clientId [" + clientId + "], clientSecret [" + clientSecret + "], applicationAccessToken [" + applicationAccessToken + "]," +
                " mercadoPagoId (code) [" + mercadoPagoId + "], redirectUri [" + redirectUri + "], refreshToken [" + refreshToken +"]");

        boolean isSandbox = StringUtils.equalsIgnoreCase(environment, "sandbox");

        Form form = new Form();
        if (isSandbox) {
            //TEST
            form.param("client_secret", applicationAccessToken);
            form.param("code", mercadoPagoId);
            form.param("redirect_uri", redirectUri);
            form.param("grant_type", "authorization_code");

        } else {
            //PROD
            form.param("client_id", clientId);
            form.param("client_secret", clientSecret);
            form.param("code", mercadoPagoId);
            form.param("redirect_uri", redirectUri);
            form.param("grant_type", "authorization_code");
        }

        //Si vino el refresh token, cambio el grant_type para la operacion de refresh
        if (StringUtils.isNotEmpty(refreshToken)) {
            form.param("refresh_token", refreshToken);
            form.param("grant_type", "authorization_code");
        }

        MultivaluedMap<String, String> formMap = form.asMap();
        String formMapAsString = formMap.keySet().stream().map(key -> key + "=" + formMap.get(key)).collect(Collectors.joining(", ", "{", "}"));

        log.debug("[getAuthorizationCodeToken] Por recuperar token con isSandobx [" + isSandbox + "], form [" + formMapAsString + "]");
        MPTokenOAuth2 tokenOAuth2Response = getConnector().genericPost(form, MPTokenOAuth2.class, MPErrorResponse.class, "/oauth/token", MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
        log.debug("[getAuthorizationCodeToken] Token recuperado [" + tokenOAuth2Response + "]");
        return tokenOAuth2Response;

    }

    /*
        Parece que no se usa
     */
    public User getUsersMe(String userAccessToken) throws RestConnectorException {
        log.debug("[getUsersMe] Por recuperar usuario con userAccessToken [" + userAccessToken + "]");
        QueryBuilder queryBuilder = QueryBuilder.Builder().path("/users/me").add("access_token", userAccessToken);
        User user = getConnector().genericGet(Void.class, User.class, MPErrorResponse.class, queryBuilder.build());
        log.debug("[getUsersMe] User recuperado [" + user + "]");
        return user;
    }

    public Payment postPayments(Payment payment) throws RestConnectorException {
        log.debug("[postPayments] Por enviar el pago [" + payment + "]");
        QueryBuilder queryBuilder = QueryBuilder.Builder().path("/v1/payments").add("access_token", accessToken);
        Payment paymentResponse = getConnector().genericPost(payment, Payment.class, MPErrorResponse.class, queryBuilder.build());
        log.debug("[postPayments] Payment devuelto [" + paymentResponse + "]");
        return paymentResponse;
    }

    public MercadoPagoReportResponse postMercadoPagoReport(MercadoPagotReportRequest request, String reportType) throws RestConnectorException {
        log.debug("[postMercadoPagoReport] Request POST mercadoPago");
        MercadoPagoReportResponse response =  getConnector().genericPost(request, MercadoPagoReportResponse.class,"/v1/account/" + reportType + "?access_token=" + accessToken);
        log.debug("[postMercadoPagoReport] Response [" + response + "]");
        return response;
    }

    public List<MercadoPagoReportResponse> getMercadoPagoReports(String reportType) throws RestConnectorException{
        log.debug("[getMercadoPagoReports] Request GET getMercadoPagoReports");
        MercadoPagoReportResponse[] responseArray = getConnector().genericGet(null, MercadoPagoReportResponse[].class, "/v1/account/"+ reportType + "/list?access_token=" + accessToken);
        List<MercadoPagoReportResponse> response = Arrays.asList(responseArray);
        log.debug("[getMercadoPagoReports] Request GET getMercadoPagoReports response [" + StringUtil.toDebugString(response) + "]");
        return response;
    }

    public String downloadMercadoPagoReport(String fileName, String reportType) throws RestConnectorException{
        log.debug("[downloadMercadoPagoReport] Request GET bank_report fileName [" + fileName + "]");
        String response = getConnector().genericGet(null, String.class, "/v1/account/" + reportType + "/" + fileName + "?access_token=" + accessToken);
        log.debug("[downloadMercadoPagoReport] Response con fileName [" + fileName + "]" + "[" + response + "]");
        return response;
    }

    public MercadoPagoReportConfiguration getMercadoPagoReportConfiguration(String reportType) throws RestConnectorException {
        log.debug("[getMercadoPagoReportConfiguration] Request para el reportType [" + reportType + "]");
        QueryBuilder queryBuilder = QueryBuilder.Builder().path("/v1/account/" + reportType + "/config").add("access_token", accessToken);
        MercadoPagoReportConfiguration responseConfiguration = getConnector().genericGet(Void.class, MercadoPagoReportConfiguration.class, MPErrorReport.class, queryBuilder.build());
        log.debug("[getMercadoPagoReportConfiguration] Response [" + responseConfiguration + "]");
        return responseConfiguration;
    }

    public MercadoPagoReportConfiguration postMercadoPagoReportConfiguration(MercadoPagoReportConfiguration requestConfiguration, String reportType) throws RestConnectorException {
        log.debug("[postMercadoPagoReportConfiguration] Request para el reportType [" + reportType + "], con configuracion [" + requestConfiguration + "]");
        QueryBuilder queryBuilder = QueryBuilder.Builder().path("/v1/account/" + reportType + "/config").add("access_token", accessToken);
        MercadoPagoReportConfiguration responseConfiguration = getConnector().genericPost(requestConfiguration, MercadoPagoReportConfiguration.class, MPErrorReport.class, queryBuilder.build());
        log.debug("[postMercadoPagoReportConfiguration] Response [" + responseConfiguration + "]");
        return responseConfiguration;

    }

    public MercadoPagoReportConfiguration putMercadoPagoReportConfiguration(MercadoPagoReportConfiguration requestConfiguration, String reportType) throws RestConnectorException {
        log.debug("[putMercadoPagoReportConfiguration] Request para el reportType [" + reportType + "], con configuracion [" + requestConfiguration + "]");
        QueryBuilder queryBuilder = QueryBuilder.Builder().path("/v1/account/" + reportType + "/config").add("access_token", accessToken);
        MercadoPagoReportConfiguration responseConfiguration = getConnector().genericPut(requestConfiguration, MercadoPagoReportConfiguration.class, MPErrorReport.class, queryBuilder.build());
        log.debug("[putMercadoPagoReportConfiguration] Response [" + responseConfiguration + "]");
        return responseConfiguration;
    }


    private RestConnector getConnector() {
        RestConnectorEnvironment environment = new RestConnectorEnvironment();
        environment.baseUrl = mpBaseUrl;
        return new RestConnector(environment,this);
    }

    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        builder.header("Authorization", "Bearer " + accessToken);
        return builder;
    }

    @Override
    public boolean getDisableHTTPSErrors() {
        return false;
    }
    @Override
    public boolean retryOnUnauthorized() {return false;}



}
