package ar.com.sdd.nacionapi.core;

import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import ar.com.sdd.commons.rest.model.TokenOAuth2;
import ar.com.sdd.commons.util.SimpleCache;
import ar.com.sdd.commons.util.SimpleCacheManager;
import ar.com.sdd.commons.util.ThreadUtil;
import ar.com.sdd.nacionapi.io.*;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.Form;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

@SuppressWarnings("unused")
public class NacionApiConnector implements RestSecurityManager {

    private final static Logger log = LogManager.getLogger(NacionApiConnector.class);

    private final RestConnector restConnector;
    private final String accessTokenBaseUrl;
    private final String clientId;
    private final String username;
    private final String password;

    private final static int MAX_RETRIES = 3;
    private final static int RETRY_WAIT_MILIS = 200;

    public NacionApiConnector(NacionApiConnectorContext context) {
        this.accessTokenBaseUrl = context.getAccessTokenBaseUrl();
        this.clientId = context.getClientId();
        this.username = context.getUsername();
        this.password = context.getPassword();

        log.trace("Creando NacionApiConnector para url [{}]", context.getBaseUrl());

        RestConnectorEnvironment environment = new RestConnectorEnvironment(context.getBaseUrl());
        restConnector = new RestConnector(environment, this);
    }

    public GenerateHTMLFormResponse generateHTMLForm(GenerateHTMLFormRequest request) throws RestConnectorException {
        final String path = "/formularioAutorizacion";
        log.debug("[generateHTMLForm] Request POST por pedir el formulario HTML al banco, request [{}] path [{}]", request, path);
        return restConnector.genericPost(request, GenerateHTMLFormResponse.class, CommonResponse.class, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
    }

    public GetPromotionsResponse getPromotions(GetPromotionsRequest request) throws RestConnectorException {
        final String path = "/obtenerPromociones";
        log.debug("[getPromotions] Request POST por obtener promociones, request [{}] path [{}]", request, path);
        return restConnector.genericPost(request, GetPromotionsResponse.class, CommonResponse.class, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
    }

    public AuthorizeTransactionResponse authorizeTransaction(AuthorizeTransactionRequest request) throws RestConnectorException {
        final String path = "/autorizar";
        log.debug("[authorizeTransaction] Request POST por autorizar transaccion, request [{}] path [{}]", request, path);
        return restConnector.genericPost(request, AuthorizeTransactionResponse.class, CommonResponse.class, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
    }

    public TransactionStatusResponse getTransactionStatus(TransactionStatusRequest request) throws RestConnectorException {
        final String path = "/consultaTransaccion";
        log.debug("[getTransactionStatus] Request POST por consultar el estado de una transaccion, request [{}] path [{}]", request, path);
        return restConnector.genericPost(request, TransactionStatusResponse.class, CommonResponse.class, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
    }

    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        if (builder != null) {
            builder.header("Authorization", "Bearer " + getAccessToken(false));
        }
        return builder;
    }

    @Override
    public boolean getDisableHTTPSErrors() {
        return false;
    }

    @Override
    public boolean retryOnUnauthorized() {
        try {
            getAccessToken(true);
            return true;
        } catch (RestConnectorException e) {
            log.error(e);
            return false;
        }
    }

    public String getAccessToken(boolean force) throws RestConnectorException {
        SimpleCache cache = SimpleCacheManager.getInstance().getCache(NacionApiConnector.class.getName());
        String cacheKey = SimpleCacheManager.buildKey("accessToken");
        String accessToken = (String) cache.get(cacheKey);
        if (force || accessToken == null) {
            Form form = new Form();
            form.param("client_id", clientId);
            form.param("username", username);
            form.param("password", password);
            form.param("grant_type", "password");

            log.debug("[getAccessToken] Por recuperar token con client_id [{}], username [{}] y password [{}]", clientId, username, password);

            RestConnector connector = new RestConnector(new RestConnectorEnvironment(accessTokenBaseUrl), null);
            TokenOAuth2 tokenOAuth2Response = connector.genericPost(form, TokenOAuth2.class, "/auth/realms/WebTarjeta/protocol/openid-connect/token", MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);

            log.debug("[getAccessToken] Token recuperado [{}]. Expira en [{}] segs. Lo agrego al a cache", tokenOAuth2Response, tokenOAuth2Response.getExpiresIn());
            accessToken = tokenOAuth2Response.getAccessToken();
            cache.put(cacheKey, accessToken, tokenOAuth2Response.getExpiresIn() - 10);
        }

        return accessToken;
    }
}