package ar.com.sdd.patagoniaapi.core;

import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import ar.com.sdd.commons.rest.model.TokenOAuth2;
import ar.com.sdd.patagoniaapi.io.LoginRequest;
import ar.com.sdd.patagoniaapi.io.LoginResponse;
import ar.com.sdd.patagoniaapi.model.Documento;
import ar.com.sdd.patagoniaapi.model.Empresa;
import ar.com.sdd.patagoniaapi.model.Usuario;
import ar.com.sdd.patagoniaapi.util.SimpleCache;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.log4j.Logger;

import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.Form;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import java.time.temporal.ChronoUnit;

public class PatagoniaApiConnector implements RestSecurityManager {

    private final static Logger log = Logger.getLogger(PatagoniaApiConnector.class);

    private final RestConnector restConnector;

    // El "BP" del final es porque es Banco Patagonia, pero es variable. En este contexto no vale la pena parametrizarlo
    private final String GET_TOKEN_URL = "https://no-prod-api.bancopatagonia.com.ar/qa/empresas/v2/login/ch/BP";
    private final String baseUrl;
    private final String xUid;
    private final String usuarioAlias;
    private final String documentoTipo;
    private final String documentoNumero;

    private final SimpleCache<String, String> cache;

    public PatagoniaApiConnector(PatagoniaApiConnectorContext context) {
        this.baseUrl = context.getBaseUrl();
        this.xUid = context.getXUid();
        this.usuarioAlias = context.getUsuarioAlias();
        this.documentoTipo = context.getDocumentoTipo();
        this.documentoNumero = context.getDocumentoNumero();

        log.debug("Creando PatagoniaApiConnector con " + context);
        RestConnectorEnvironment environment = new RestConnectorEnvironment(baseUrl);
        restConnector = new RestConnector(environment, this);

        // Inicializo la cache que va a contener el accessToken. El mismo expira luego de 1 hora
        cache = SimpleCache.builder()
                .maximumSize(1)                                  // Puedo tener solo 1 token activo
                .expireAfter(3599, ChronoUnit.MINUTES) // El token expira luego de 1 hora
                .build();
    }

    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        if (builder != null) {
            builder.header("x-uid", xUid);
            builder.header("", getAccessToken());
        }
        return builder;
    }

    @Override
    public boolean getDisableHTTPSErrors() {
        return false;
    }

    public String getAccessToken() throws RestConnectorException {
        String accessToken = cache.getIfPresent("accessToken");
        if (accessToken == null) {
            Usuario usuario = new Usuario();
            usuario.setAlias(usuarioAlias);

            Documento documento = new Documento();
            documento.setTipo(documentoTipo);
            documento.setNumero(documentoNumero);

            Empresa empresa = new Empresa();
            empresa.setDocumento(documento);
            LoginRequest request = new LoginRequest();
            request.setUsuario(usuario);

            log.debug("[getAccessToken] Por recuperar token con [" + request + "]");

            RestConnector connector = new RestConnector(new RestConnectorEnvironment(GET_TOKEN_URL), null);
            Pair<LoginResponse, MultivaluedMap<String, Object>> responsePair = connector.genericPostWithResponseHeader(request, LoginResponse.class, null, "", MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON);
            LoginResponse loginResponse = responsePair.getKey();
            log.info(loginResponse);
            MultivaluedMap<String, Object> headers = responsePair.getValue();
            log.info("HEADERS");
            for (String headerName : headers.keySet()) {
                log.info(headerName + "=" + headers.get(headerName));
            }
//            log.debug("[getAccessToken] Token recuperado [" + tokenOAuth2Response + "]. Lo agrego al a cache");
//            accessToken = tokenOAuth2Response.getAccessToken();
            cache.put("accessToken", accessToken);
        }

        return accessToken;
    }
}
