package ar.com.sdd.patagoniaapi.core;

import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import ar.com.sdd.patagoniaapi.io.*;
import ar.com.sdd.patagoniaapi.model.*;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.*;

public class PatagoniaApiConnector implements RestSecurityManager {

    private final static Logger log = LogManager.getLogger(PatagoniaApiConnector.class);

    private final RestConnector restConnector;
    private final String xApplicationId;
    private final String usuarioAlias;
    private final String documentoTipo;
    private final String documentoNumero;
    private final String adherente;
    private final String convenio;
    private final String docProd;
    private boolean doingLogin;
    private String accessToken;
    private final static String channel = "PT";

    public final static String STATUS_OK = "OK";
    public final static String STATUS_ERROR = "ERROR";

    public PatagoniaApiConnector(PatagoniaApiConnectorContext context) {
        this.usuarioAlias = context.getUsuarioAlias();
        this.documentoTipo = context.getDocumentoTipo();
        this.documentoNumero = context.getDocumentoNumero();
        this.xApplicationId = context.getXApplicationKey();
        this.adherente = context.getAdherente();
        this.convenio = context.getConvenio();
        this.docProd = context.getDocProd();

        log.debug("Creando PatagoniaApiConnector con " + context);
        RestConnectorEnvironment environment = new RestConnectorEnvironment(context.getBaseUrl());
        restConnector = new RestConnector(environment, this);
    }

    public EmpresaCobroArchivoEstadoResponse publicarDeuda(String fileName, String fileInBase64) throws RestConnectorException {
        log.debug("[uploadFile] Por hacer upload del archivo");

        // Genero el baerer token
        generateAccessToken();

        EmpresaArchivoRequest empresaArchivoRequest = new EmpresaArchivoRequest();
        empresaArchivoRequest.setToken(generateUploadToken());
        empresaArchivoRequest.setUserId("B2B-" + documentoNumero + "-0");
        empresaArchivoRequest.setFileName(fileName);
        empresaArchivoRequest.setFile(fileInBase64);

        EmpresaArchivoResponse empresaArchivoResponse = restConnector.genericPost(empresaArchivoRequest, EmpresaArchivoResponse.class, "/archivo", MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
        log.debug("[uploadFile] Response: " + empresaArchivoResponse);
        if (empresaArchivoResponse != null && empresaArchivoResponse.getRespuesta() != null) {
            Respuesta respuesta = empresaArchivoResponse.getRespuesta();
            if (respuesta.getCodigo().equals("OK")) {
                return confirmarOperacion(empresaArchivoRequest.getToken(), fileName.substring(0, 2));
            }
        }

        return null;
    }

    public EmpresaCobroArchivoEstadoResponse confirmarOperacion(String token, String tipoArchivo) throws RestConnectorException {
        log.debug("[confirmarOperacion] Por confirmar procesamiento del archivo que acabamos de subir");
        EmpresaCobroArchivoEstadoRequest empresaCobroArchivoEstadoRequest = new EmpresaCobroArchivoEstadoRequest();
        empresaCobroArchivoEstadoRequest.setToken(token);

        final String subconvenio = "1";
        String confirmPath = "/cobro/archivo/prod/" + docProd + "/adherentes/" + adherente + "/" + convenio + "/" + subconvenio + "/estado/tipo-arch/" + tipoArchivo + "/doc-tp/" + documentoTipo + "/docNum/" + documentoNumero + "/ch/" + channel;
        return restConnector.genericPost(empresaCobroArchivoEstadoRequest, EmpresaCobroArchivoEstadoResponse.class, ErrorResponse.class, confirmPath);
    }

    public void consultaConvenioPorAdherente() {
        // 5 - Metodo no implementado - ver documentacion - No nos aporta nada
    }

    public EmpresaCobroListadoResponse consultarPosicionIntegralDeCobros(Date fromDate, Date toDate) throws RestConnectorException {
        generateAccessToken();

        final SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd");
        Map<String, Object> params = new HashMap<>();
        if (fromDate != null) {
            params.put("fch-rec-dsd", sdf.format(fromDate));
        }
        if (toDate != null) {
            params.put("fch-rec-hst", sdf.format(toDate));
        }
        String url = "/cobro/prod/" + docProd + "/adherentes/" + adherente + "/" + convenio + "/1/ch/" + channel;
        return restConnector.genericGet(null, EmpresaCobroListadoResponse.class, ErrorResponse.class, url, params);
    }

    /**
     * @param pagCant Cantidad de elementos por pagina del paginador
     * @param pagAct  Numero de pagina del paginador
     */
    public EmpresaCobroDetalleResponse consultarDetallePosicionIntegralDeCobros(String subconvenio, String numeroCanal, String numeroConsulta, Date fechaRecuperoDesde, Date fechaRecuperoHasta, String deudor1, String deudor2,
                                                                                String comprobante1, String comprobante2, String comprobante3, String comprobante4, String comprobante5, int pagCant, int pagAct) throws RestConnectorException{
        final SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd");
        Map<String, Object> params = new HashMap<>();
        if (fechaRecuperoDesde != null) {
            params.put("fch-rec-dsd", sdf.format(fechaRecuperoDesde));
        }
        if (fechaRecuperoHasta != null) {
            params.put("fch-rec-hst", sdf.format(fechaRecuperoHasta));
        }
        if (StringUtils.isNotEmpty(deudor1)) {
            params.put("deudor1", deudor1);
        }
        if (StringUtils.isNotEmpty(deudor2)) {
            params.put("deudor2", deudor2);
        }
        if (StringUtils.isNotEmpty(comprobante1)) {
            params.put("comprobante1", comprobante1);
        }
        if (StringUtils.isNotEmpty(comprobante2)) {
            params.put("comprobante2", comprobante2);
        }
        if (StringUtils.isNotEmpty(comprobante3)) {
            params.put("comprobante3", comprobante3);
        }
        if (StringUtils.isNotEmpty(comprobante4)) {
            params.put("comprobante4", comprobante4);
        }
        if (StringUtils.isNotEmpty(comprobante5)) {
            params.put("comprobante5", comprobante5);
        }
        params.put("pag-cant", pagCant);
        params.put("pag-act", pagAct);
        String url = "/cobro/prod/" + docProd + "/adherentes/"  + adherente + "/" + convenio + "/"  + subconvenio + "/canal/" + numeroCanal + "/consulta/" + numeroConsulta;
        return restConnector.genericGet(null, EmpresaCobroDetalleResponse.class, ErrorResponse.class, url, params);
    }

    public void historialPublicacionDeArchivos() {
        // 8 - Metodo no implementado - ver documentacion - No nos aporta nada
    }

    public void consultaArchivosRecepcion() {
        // 9 - Metodo no implementado - ver documentacion - No nos aporta nada
    }

    public void consultaEstadoArchivo() {
        // 10 - Se usa para consultar los getCollectionSetStatus y postCollectionSetStatus
    }

    public void consultaErroresDelArchivo() {
        // 11
    }

    public void generarTokenParaBajarArchivoDeCobranzas(){
        // 12 - Implementar
    }

    public void bajarArchivoDeCobranzas() {
        // 13 - Implementar
    }


    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        if (builder != null) {
            builder.header("x-application-key", xApplicationId);
            if (!doingLogin) builder.header("Authorization", "Bearer " + accessToken);
        }
        return builder;
    }

    @Override
    public boolean getDisableHTTPSErrors() {
        return false;
    }

    public void generateAccessToken() throws RestConnectorException {
        // Construyo el LoginRequest
        Usuario usuario = new Usuario();
        usuario.setAlias(usuarioAlias);

        Documento documento = new Documento();
        documento.setTipo(documentoTipo);
        documento.setNumero(documentoNumero);

        Empresa empresa = new Empresa();
        empresa.setDocumento(documento);

        LoginRequest request = new LoginRequest();
        request.setUsuario(usuario);
        request.setEmpresa(empresa);

        log.debug("[loginAndGetAccessToken] Por hacer login y obtener token con [" + request + "]");

        // Marca para que no agregue en el header "Authorization"
        doingLogin = true;
        Pair<LoginResponse, MultivaluedMap<String, Object>> responsePair = restConnector.genericPostWithResponseHeader(request, LoginResponse.class, "/v2/login/ch/" + channel);
        MultivaluedMap<String, Object> headers = responsePair.getValue();
        String xUid = null;
        List<Object> xUidList = headers.get("x-uid");
        if (xUidList != null && !xUidList.isEmpty()) {
            xUid = (String) xUidList.get(0);
        }

        if (xUid == null || xUid.trim().isEmpty()) {
            throw new RuntimeException("No se pudo obtener el parametro [x-uid] del response header");
        }

        String xAccessToken = null;
        List<Object> xAccessTokenList = headers.get("x-access-token");
        if (xAccessTokenList != null && !xAccessTokenList.isEmpty()) {
            xAccessToken = (String) xAccessTokenList.get(0);
        }

        if (xAccessToken == null || xAccessToken.trim().isEmpty()) {
            throw new RuntimeException("No se pudo obtener el parametro [x-access-token] del response header");
        }

        accessToken = Base64.getEncoder().encodeToString((xUid + ":" + xAccessToken).getBytes(StandardCharsets.UTF_8));

        log.debug("[loginAndGetAccessToken] Token recuperado y codificado a base64 [" + accessToken + "]. Lo agrego al a cache");

        doingLogin = false;
    }

    public String generateUploadToken() throws RestConnectorException {
        log.debug("[generateUploadToken] Por obtener el token para subir el archivo");

        EmpresaArchivoTokenResponse empresaArchivoTokenResponse = restConnector.genericPost(new EmpresaArchivoTokenRequest(), EmpresaArchivoTokenResponse.class, "/archivo/token/sistema/CASH/accion/SUBIR/ch/" + channel);
        String uploadToken = null;
        List<Token> tokens = empresaArchivoTokenResponse.getToken();
        if (tokens != null && !tokens.isEmpty()) {
            uploadToken = tokens.get(0).getToken();
        }

        if (uploadToken == null || uploadToken.trim().isEmpty()) {
            throw new RuntimeException("[generateUploadToken] No se pudo obtener el parametro [tokenLista.token] del response");
        }

        log.debug("[generateUploadToken] Token para subir el archivo recuperado [" + uploadToken + "]");

        return uploadToken;
    }
}