package ar.com.sdd.patagoniaapi.core;

import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import ar.com.sdd.patagoniaapi.io.*;
import ar.com.sdd.patagoniaapi.model.Documento;
import ar.com.sdd.patagoniaapi.model.Empresa;
import ar.com.sdd.patagoniaapi.model.TokenLista;
import ar.com.sdd.patagoniaapi.model.Usuario;
import ar.com.sdd.patagoniaapi.util.SimpleCache;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.log4j.Logger;

import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import java.nio.charset.StandardCharsets;
import java.time.temporal.ChronoUnit;
import java.util.Base64;
import java.util.List;

public class PatagoniaApiConnector implements RestSecurityManager {

    private final static Logger log = Logger.getLogger(PatagoniaApiConnector.class);

    private final RestConnector restConnector;
    private final String xApplicationId;
    private final String usuarioAlias;
    private final String documentoTipo;
    private final String documentoNumero;

    private boolean doingLogin = false;

    //private final String OPERATION_CONFIRMATION_PATH = "/archivo/prod/30563598111/adherentes/221/102/1/estado/tipo-arch/RF/doc-tp/CUIT/docNum/30563598111/ch/BP";

    private final SimpleCache<String, String> cache;

    public PatagoniaApiConnector(PatagoniaApiConnectorContext context) {
        this.usuarioAlias = context.getPatagoniaApiUsuarioAlias();
        this.documentoTipo = context.getPatagoniaApiDocumentoTipo();
        this.documentoNumero = context.getPatagoniaApiDocumentoNumero();
        this.xApplicationId = context.getPatagoniaApiXApplicationKey();

        log.debug("Creando PatagoniaApiConnector con " + context);
        RestConnectorEnvironment environment = new RestConnectorEnvironment(context.getPatagoniaApiBaseUrl());
        restConnector = new RestConnector(environment, this);

        // Inicializo la cache que va a contener el accessToken. El mismo expira luego de 1 hora
        cache = SimpleCache.builder()
                .maximumSize(1)                                  // Puedo tener solo 1 token activo
                .expireAfter(1, ChronoUnit.MINUTES)    // El token expira luego de 1 minuto
                .build();
    }

    public FileUploadResponse publicarDeuda(String fileName, String fileInBase64) throws RestConnectorException {
        log.debug("[uploadFile] Por hacer upload del archivo");

        FileUploadRequest fileUploadRequest = new FileUploadRequest();
        fileUploadRequest.setToken(generateUploadToken());
        fileUploadRequest.setUserId("B2B-" + documentoNumero + "-0");
        fileUploadRequest.setFileName(fileName);
        fileUploadRequest.setFile(fileInBase64);

        return restConnector.genericPost(fileUploadRequest, FileUploadResponse.class, "/archivo", MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
    }

    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        if (builder != null) {
            builder.header("x-application-key", xApplicationId);
            if (!doingLogin) builder.header("Authorization", "Bearer " + getAccessToken());
        }
        return builder;
    }

    @Override
    public boolean getDisableHTTPSErrors() {
        return false;
    }

    public String getAccessToken() throws RestConnectorException {
        String accessToken = cache.getIfPresent("accessToken");
        if (accessToken == null) {
            doingLogin = true;

            // Construyo el LoginRequest
            Usuario usuario = new Usuario();
            usuario.setAlias(usuarioAlias);

            Documento documento = new Documento();
            documento.setTipo(documentoTipo);
            documento.setNumero(documentoNumero);

            Empresa empresa = new Empresa();
            empresa.setDocumento(documento);

            LoginRequest request = new LoginRequest();
            request.setUsuario(usuario);
            request.setEmpresa(empresa);

            log.debug("[loginAndGetAccessToken] Por hacer login y obtener token con [" + request + "]");

            Pair<LoginResponse, MultivaluedMap<String, Object>> responsePair = restConnector.genericPostWithResponseHeader(request, LoginResponse.class, "/v2/login/ch/PT");
            MultivaluedMap<String, Object> headers = responsePair.getValue();
            String xUid = null;
            List<Object> xUidList = headers.get("x-uid");
            if (xUidList != null && !xUidList.isEmpty()) {
                xUid = (String) xUidList.get(0);
            }

            if (xUid == null || xUid.trim().length() == 0) {
                throw new RuntimeException("No se pudo obtener el parametro [x-uid] del response header");
            }

            String xAccessToken = null;
            List<Object> xAccessTokenList = headers.get("x-access-token");
            if (xAccessTokenList != null && !xAccessTokenList.isEmpty()) {
                xAccessToken = (String) xAccessTokenList.get(0);
            }

            if (xAccessToken == null || xAccessToken.trim().length() == 0) {
                throw new RuntimeException("No se pudo obtener el parametro [x-access-token] del response header");
            }

            accessToken = Base64.getEncoder().encodeToString((xUid + ":" + xAccessToken).getBytes(StandardCharsets.UTF_8));

            log.debug("[loginAndGetAccessToken] Token recuperado y codificado a base64 [" + accessToken + "]. Lo agrego al a cache");
            cache.put("accessToken", accessToken);

            doingLogin = false;
        }

        return accessToken;
    }

    public String generateUploadToken() throws RestConnectorException {
        log.debug("[generateUploadToken] Por obtener el token para subir el archivo");

        GenerateUploadTokenResponse generateUploadTokenResponse = restConnector.genericPost(new GenerateUploadTokenRequest(), GenerateUploadTokenResponse.class, "/archivo/token/sistema/CASH/accion/SUBIR/ch/PT");
        String uploadToken = null;
        List<TokenLista> tokens = generateUploadTokenResponse.getTokenLista();
        if (tokens != null && !tokens.isEmpty()) {
            uploadToken = tokens.get(0).getToken();
        }

        if (uploadToken == null || uploadToken.trim().length() == 0) {
            throw new RuntimeException("[generateUploadToken] No se pudo obtener el parametro [tokenLista.token] del response");
        }

        log.debug("[generateUploadToken] Token para subir el archivo recuperado [" + uploadToken + "]");

        return uploadToken;
    }
}