package ar.com.sdd.patagoniaapi.core;

import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import ar.com.sdd.patagoniaapi.io.*;
import ar.com.sdd.patagoniaapi.model.*;
import ar.com.sdd.patagoniaapi.util.SimpleCache;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.log4j.Logger;

import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import java.nio.charset.StandardCharsets;
import java.time.temporal.ChronoUnit;
import java.util.Base64;
import java.util.Date;
import java.util.List;

public class PatagoniaApiConnector implements RestSecurityManager {

    private final static Logger log = Logger.getLogger(PatagoniaApiConnector.class);

    private final RestConnector restConnector;
    private final String xApplicationId;
    private final String usuarioAlias;
    private final String documentoTipo;
    private final String documentoNumero;
    private final String adherente;
    private final String convenio;

    private final static String channel = "PT";
    private boolean doingLogin = false;

    private final SimpleCache<String, String> cache;

    private final static String STATUS_OK = "OK";
    private final static String STATUS_ERROR = "ERROR";

    public PatagoniaApiConnector(PatagoniaApiConnectorContext context) {
        this.usuarioAlias = context.getPatagoniaApiUsuarioAlias();
        this.documentoTipo = context.getPatagoniaApiDocumentoTipo();
        this.documentoNumero = context.getPatagoniaApiDocumentoNumero();
        this.xApplicationId = context.getPatagoniaApiXApplicationKey();
        this.adherente = context.getPatagoniaApiAdherente();
        this.convenio = context.getPatagoniaApiConvenio();

        log.debug("Creando PatagoniaApiConnector con " + context);
        RestConnectorEnvironment environment = new RestConnectorEnvironment(context.getPatagoniaApiBaseUrl());
        restConnector = new RestConnector(environment, this);

        // Inicializo la cache que va a contener el accessToken. El mismo expira luego de 1 hora
        cache = SimpleCache.builder()
                .maximumSize(1)                                  // Puedo tener solo 1 token activo
                .expireAfter(1, ChronoUnit.MINUTES)    // El token expira luego de 1 minuto
                .build();
    }

    public EmpresaCobroArchivoEstadoResponse publicarDeuda(String fileName, String fileInBase64) throws RestConnectorException {
        log.debug("[uploadFile] Por hacer upload del archivo");

        EmpresaArchivoRequest empresaArchivoRequest = new EmpresaArchivoRequest();
        empresaArchivoRequest.setToken(generateUploadToken());
        empresaArchivoRequest.setUserId("B2B-" + documentoNumero + "-0");
        empresaArchivoRequest.setFileName(fileName);
        empresaArchivoRequest.setFile(fileInBase64);

        EmpresaArchivoResponse empresaArchivoResponse = restConnector.genericPost(empresaArchivoRequest, EmpresaArchivoResponse.class, "/archivo", MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
        log.debug("[uploadFile] Response: " + empresaArchivoResponse);
        if (empresaArchivoResponse != null && empresaArchivoResponse.getRespuesta() != null) {
            Respuesta respuesta = empresaArchivoResponse.getRespuesta();
            if (respuesta.getCodigo().equals("OK")) {
                return confirmarOperacion(empresaArchivoRequest.getToken(), fileName.substring(0, 2));
            }
        }

        return null;
    }

    public EmpresaCobroArchivoEstadoResponse confirmarOperacion(String token, String tipoArchivo) throws RestConnectorException {
        log.debug("[confirmarOperacion] Por confirmar procesamiento del archivo que acabamos de subir");
        EmpresaCobroArchivoEstadoRequest empresaCobroArchivoEstadoRequest = new EmpresaCobroArchivoEstadoRequest();
        empresaCobroArchivoEstadoRequest.setToken(token);

        // TODO: Preguntar como se llenan estos valores
        final String docProd = documentoNumero;
        final String subconvenio = "1";
        String confirmPath = "/archivo/prod/" + docProd + "/adherentes/" + adherente + "/" + convenio + "/" + subconvenio + "/estado/tipo-arch/" + tipoArchivo + "/doc-tp/" + documentoTipo + "/docNum/" + documentoNumero + "/ch/" + channel;
        return restConnector.genericPost(empresaCobroArchivoEstadoRequest, EmpresaCobroArchivoEstadoResponse.class, confirmPath);
    }

    public void consultaConvenioPorAdherente() {
        // 5 - Metodo no implementado - ver documentacion - No nos aporta nada
    }

    public void postPaymentSet(String archivoId) throws RestConnectorException {
        String docProd = documentoNumero;
        EmpresaCobroListadoResponse empresaCobroListadoResponse = consultarPosicionIntegralDeCobros(docProd, archivoId, new Date(), new Date());
        log.info(empresaCobroListadoResponse);
        if (empresaCobroListadoResponse != null && empresaCobroListadoResponse.getEstadoEjecucion().equalsIgnoreCase(STATUS_OK)) {
            for (Recaudacion recaudacion : empresaCobroListadoResponse.getRecaudacion()) {
                for (ListaCobros cobros : recaudacion.getListaCobros()) {
                    consultarDetallePosicionIntegralDeCobros(docProd, cobros.getSubconvenioId(), recaudacion.getCanal(), recaudacion.getNumeroConsulta());
                }
            }
        }
    }

    public EmpresaCobroListadoResponse consultarPosicionIntegralDeCobros(String docProd, String archivoId, Date fromDate, Date toDate) throws RestConnectorException {
        EmpresaCobroListadoRequest empresaCobroListadoRequest = new EmpresaCobroListadoRequest();
        empresaCobroListadoRequest.setFechaRecuperoDesde(fromDate);
        empresaCobroListadoRequest.setFechaRecuperoDesde(toDate);
        String url = "/archivo/estado/prod/" + docProd + "/adherentes/" + adherente + "/" + archivoId + "/ch/" + channel;
        return restConnector.genericPost(empresaCobroListadoRequest, EmpresaCobroListadoResponse.class, url);
    }

    public void consultarDetallePosicionIntegralDeCobros(String docProd, String subconvenio, String numeroCanal, String numeroConsulta) {
        String url = "/cobro/prod/" + docProd + "/adherentes/"  + adherente + "/" + convenio + "/"  + subconvenio + "/canal/" + numeroCanal + "/consulta/" + numeroConsulta;
        log.info("por consultar: " + url);
    }

    public void historialPublicacionDeArchivos() {
        // 8 - Metodo no implementado - ver documentacion - No nos aporta nada
    }

    public void consultaArchivosRecepcion() {
        // 9 - Metodo no implementado - ver documentacion - No nos aporta nada
    }

    public void consultaEstadoArchivo() {
        // 10 - Se usa para consultar los getCollectionSetStatus y postCollectionSetStatus
    }

    public void consultaErroresDelArchivo() {
        // 11
    }

    public void generarTokenParaBajarArchivoDeCobranzas(){
        // 12 - Implementar
    }

    public void bajarArchivoDeCobranzas() {
        // 13 - Implementar
    }


    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        if (builder != null) {
            builder.header("x-application-key", xApplicationId);
            if (!doingLogin) builder.header("Authorization", "Bearer " + getAccessToken());
        }
        return builder;
    }

    @Override
    public boolean getDisableHTTPSErrors() {
        return false;
    }

    public String getAccessToken() throws RestConnectorException {
        String accessToken = cache.getIfPresent("accessToken");
        if (accessToken == null) {
            doingLogin = true;

            // Construyo el LoginRequest
            Usuario usuario = new Usuario();
            usuario.setAlias(usuarioAlias);

            Documento documento = new Documento();
            documento.setTipo(documentoTipo);
            documento.setNumero(documentoNumero);

            Empresa empresa = new Empresa();
            empresa.setDocumento(documento);

            LoginRequest request = new LoginRequest();
            request.setUsuario(usuario);
            request.setEmpresa(empresa);

            log.debug("[loginAndGetAccessToken] Por hacer login y obtener token con [" + request + "]");

            Pair<LoginResponse, MultivaluedMap<String, Object>> responsePair = restConnector.genericPostWithResponseHeader(request, LoginResponse.class, "/v2/login/ch/" + channel);
            MultivaluedMap<String, Object> headers = responsePair.getValue();
            String xUid = null;
            List<Object> xUidList = headers.get("x-uid");
            if (xUidList != null && !xUidList.isEmpty()) {
                xUid = (String) xUidList.get(0);
            }

            if (xUid == null || xUid.trim().length() == 0) {
                throw new RuntimeException("No se pudo obtener el parametro [x-uid] del response header");
            }

            String xAccessToken = null;
            List<Object> xAccessTokenList = headers.get("x-access-token");
            if (xAccessTokenList != null && !xAccessTokenList.isEmpty()) {
                xAccessToken = (String) xAccessTokenList.get(0);
            }

            if (xAccessToken == null || xAccessToken.trim().length() == 0) {
                throw new RuntimeException("No se pudo obtener el parametro [x-access-token] del response header");
            }

            accessToken = Base64.getEncoder().encodeToString((xUid + ":" + xAccessToken).getBytes(StandardCharsets.UTF_8));

            log.debug("[loginAndGetAccessToken] Token recuperado y codificado a base64 [" + accessToken + "]. Lo agrego al a cache");
            cache.put("accessToken", accessToken);

            doingLogin = false;
        }

        return accessToken;
    }

    public String generateUploadToken() throws RestConnectorException {
        log.debug("[generateUploadToken] Por obtener el token para subir el archivo");

        EmpresaArchivoTokenResponse empresaArchivoTokenResponse = restConnector.genericPost(new EmpresaArchivoTokenRequest(), EmpresaArchivoTokenResponse.class, "/archivo/token/sistema/CASH/accion/SUBIR/ch/" + channel);
        String uploadToken = null;
        List<Token> tokens = empresaArchivoTokenResponse.getToken();
        if (tokens != null && !tokens.isEmpty()) {
            uploadToken = tokens.get(0).getToken();
        }

        if (uploadToken == null || uploadToken.trim().length() == 0) {
            throw new RuntimeException("[generateUploadToken] No se pudo obtener el parametro [tokenLista.token] del response");
        }

        log.debug("[generateUploadToken] Token para subir el archivo recuperado [" + uploadToken + "]");

        return uploadToken;
    }
}