package ar.com.sdd.patagoniaapi.core;

import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import ar.com.sdd.commons.util.ApplicationException;
import ar.com.sdd.commons.util.SimpleCache;
import ar.com.sdd.commons.util.SimpleCacheManager;
import ar.com.sdd.commons.util.StringUtil;
import ar.com.sdd.patagoniaapi.io.*;
import ar.com.sdd.patagoniaapi.model.*;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Base64;
import java.util.List;
import java.util.Optional;

@SuppressWarnings("unused")
public abstract class PatagoniaApiConnectorBase implements RestSecurityManager {

    private final static Logger log = LogManager.getLogger(PatagoniaApiConnectorBase.class);

    protected final RestConnector restConnector;
    protected final String xApplicationId;
    protected final String usuarioAlias;
    protected final String documentoTipo;
    protected final String documentoNumero;
    protected final String adherente;
    protected final String convenio;
    protected final String docProd;
    protected final static String channel = "PT";
    private boolean doingLogin;
    private final int accessTokenTtl;
    private final SimpleCache cache = SimpleCacheManager.getInstance().getCache(PatagoniaApiConnectorBase.class.getName());

    public final static String STATUS_OK = "OK";
    public final static String STATUS_ERROR = "ERROR";

    protected final SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd");

    public PatagoniaApiConnectorBase(PatagoniaApiConnectorContext context) {
        this.usuarioAlias = context.getUsuarioAlias();
        this.documentoTipo = context.getDocumentoTipo();
        this.documentoNumero = context.getDocumentoNumero();
        this.xApplicationId = context.getXApplicationKey();
        this.adherente = context.getAdherente();
        this.convenio = context.getConvenio();
        this.docProd = context.getDocProd();
        this.accessTokenTtl = context.getAccessTokenTtl();

        log.debug("Creando PatagoniaApiConnector con " + context);
        RestConnectorEnvironment environment = new RestConnectorEnvironment(context.getBaseUrl());
        restConnector = new RestConnector(environment, this);
    }

    public EmpresaArchivoEstadoResponse uploadFile(String fileName, String fileInBase64) throws RestConnectorException {
        log.debug("[uploadFile] Por hacer upload del archivo " + fileName);

        String userId = (String) cache.get("userId");
        String accessToken = (String) cache.get("accessToken");

        if (StringUtil.isEmpty(userId) || StringUtil.isEmpty(accessToken)) {
            Pair<LoginResponse, MultivaluedMap<String, Object>> loginResponse = login();
            // Obtengo el userId del response
            userId = getUserId(loginResponse.getKey().getUsuario().getUriOperador());
            cache.put("userId", userId);
            generateAccessTokenFromResponseHeader(loginResponse.getValue());
        }

        // 2 - Genero el token para subir el archivo y luego procesarlo
        final String uploadToken = generateUploadToken();

        // 3 - Armo el request para subir el archivo
        EmpresaArchivoRequest empresaArchivoRequest = new EmpresaArchivoRequest();
        empresaArchivoRequest.setToken(uploadToken);
        empresaArchivoRequest.setUserId(userId);
        empresaArchivoRequest.setFileName(fileName);
        empresaArchivoRequest.setFile(fileInBase64);

        final String path = "/archivo";
        EmpresaArchivoResponse empresaArchivoResponse = restConnector.genericPost(empresaArchivoRequest, EmpresaArchivoResponse.class, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
        if (empresaArchivoResponse == null) {
            throw new RuntimeException("[uploadFile] El response de la subida del archivo " + fileName + " fue null");
        } else if (empresaArchivoResponse.getRespuesta() == null) {
            throw new RuntimeException("[uploadFile] No se pudo obtener el parametro [respuesta] del response");
        }

        // 4 - Si se subio ok, le indico que lo procesen
        Respuesta respuesta = empresaArchivoResponse.getRespuesta();
        if (respuesta.getCodigo().equals(STATUS_OK)) {
            return confirmarOperacion(empresaArchivoRequest.getToken(), fileName);
        }

        return null;
    }

    private String getUserId(String uriOperador) {
        String uriSFB = uriOperador.split(";")[1];
        Optional<String> userId = Arrays.stream(uriSFB.split("#")).filter(s -> s.startsWith("B2B")).findFirst();
        if (userId.isEmpty()) {
            throw new RuntimeException("No se pudo obtener el userId del LoginResponse.Usuario.UriOperador [" + uriOperador + "]");
        }

        return userId.get();
    }

    public abstract EmpresaArchivoEstadoResponse confirmarOperacion(String token, String fileName) throws RestConnectorException;

    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        if (builder != null) {
            builder.header("x-application-key", xApplicationId);
            if (!doingLogin) builder.header("Authorization", "Bearer " + getAccessToken());
        }
        return builder;
    }

    @Override
    public boolean getDisableHTTPSErrors() {
        return false;
    }

    public Pair<LoginResponse, MultivaluedMap<String, Object>> login() throws RestConnectorException {
        // Construyo el LoginRequest
        Usuario usuario = new Usuario();
        usuario.setAlias(usuarioAlias);

        Documento documento = new Documento();
        documento.setTipo(documentoTipo);
        documento.setNumero(documentoNumero);

        Empresa empresa = new Empresa();
        empresa.setDocumento(documento);

        LoginRequest request = new LoginRequest();
        request.setUsuario(usuario);
        request.setEmpresa(empresa);

        log.debug("[login] Por hacer login con " + request);
        // Marca para que no agregue en el header "Authorization"
        doingLogin = true;
        final String path = "/v2/login/ch/" + channel;
        Pair<LoginResponse, MultivaluedMap<String, Object>> responsePair = restConnector.genericPostWithResponseHeader(request, LoginResponse.class, path);
        // Desmarco para que se agregue en el header "Authorization"
        doingLogin = false;
        return responsePair;
    }

    public String getAccessToken() throws RestConnectorException {
        String accessToken = (String) cache.get("accessToken");
        if (StringUtil.isEmpty(accessToken)) {
            Pair<LoginResponse, MultivaluedMap<String, Object>> loginResponse = login();
            accessToken = generateAccessTokenFromResponseHeader(loginResponse.getValue());
        }

        if (StringUtil.isEmpty(accessToken)) {
            throw new ApplicationException("Se produjo un error al intentar obtener el accessToken de la cache");
        }

        return accessToken;
    }

    protected String generateAccessTokenFromResponseHeader(MultivaluedMap<String, Object> responseHeaders) {
        String xUid = null;
        List<Object> xUidList = responseHeaders.get("x-uid");
        if (xUidList != null && !xUidList.isEmpty()) {
            xUid = (String) xUidList.get(0);
        }

        if (xUid == null || xUid.trim().isEmpty()) {
            throw new RuntimeException("[updateAccessToken] No se pudo obtener el parametro [x-uid] del response header");
        }

        String xAccessToken = null;
        List<Object> xAccessTokenList = responseHeaders.get("x-access-token");
        if (xAccessTokenList != null && !xAccessTokenList.isEmpty()) {
            xAccessToken = (String) xAccessTokenList.get(0);
        }

        if (xAccessToken == null || xAccessToken.trim().isEmpty()) {
            throw new RuntimeException("[updateAccessToken] No se pudo obtener el parametro [x-access-token] del response header");
        }

        String accessToken = Base64.getEncoder().encodeToString((xUid + ":" + xAccessToken).getBytes(StandardCharsets.UTF_8));
        log.debug("[generateAccessToken] AccessToken actualizado y codificado a base64 [" + accessToken + "]. Lo guardo en la cache");
        cache.put("accessToken", accessToken, accessTokenTtl);
        return accessToken;
    }

    public String generateUploadToken() throws RestConnectorException {
        log.debug("[generateUploadToken] Por obtener el token para subir el archivo");

        final String path = "/archivo/token/sistema/CASH/accion/SUBIR/ch/" + channel;
        EmpresaArchivoTokenResponse empresaArchivoTokenResponse = restConnector.genericPost(new EmpresaArchivoTokenRequest(), EmpresaArchivoTokenResponse.class, path);
        String uploadToken = null;
        List<Token> tokens = empresaArchivoTokenResponse.getToken();
        if (tokens != null && !tokens.isEmpty()) {
            uploadToken = tokens.get(0).getToken();
        }

        if (uploadToken == null || uploadToken.trim().isEmpty()) {
            throw new RuntimeException("[generateUploadToken] No se pudo obtener el parametro [tokenLista.token] del response");
        }

        log.debug("[generateUploadToken] Token para subir el archivo recuperado [" + uploadToken + "]");

        return uploadToken;
    }

    /**
     * 10 - Consulta de Estado del Archivo - Permite conocer el estado de un archivo
     */
    public EmpresaArchivoByIdResponse consultaEstadoArchivo(String archivoId) throws RestConnectorException {
        final String path = "/archivo/estado/prod/" + docProd + "/adherentes/" + adherente + "/" + archivoId + "/ch/" + channel;
        Pair<EmpresaArchivoByIdResponse, MultivaluedMap<String,Object>> response = restConnector.genericGetWithResponseHeader(null, EmpresaArchivoByIdResponse.class, ErrorResponse.class, path, null);
        generateAccessTokenFromResponseHeader(response.getValue());
        return response.getKey();
    }

    /**
     * 11 - Consulta de Errores del Archivo - Retorna la lista de errores de un archivo con estado ERROR
     */
    public EmpresaArchivoErroresResponse consultaErroresArchivo(String archivoId, Integer pagAct, Integer pagCant) throws RestConnectorException {
        if (pagAct == null) pagAct = 1;
        if (pagCant == null) pagCant = 300;
        final String path = "/prod/" + docProd + "/adherentes/" + adherente + "/pagos/errores-en-archivo/" + archivoId + "/ch/" + channel + "?pag-act=" + pagAct + "&pag-cant=" + pagCant;
        return restConnector.genericGet(null, EmpresaArchivoErroresResponse.class, ErrorResponse.class, path);
    }
}