package ar.com.sdd.praapi.core;

import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import ar.com.sdd.commons.rest.model.TokenOAuth2;
import ar.com.sdd.commons.util.SimpleCache;
import ar.com.sdd.commons.util.SimpleCacheManager;
import ar.com.sdd.commons.util.ThreadUtil;
import ar.com.sdd.praapi.io.*;
import ar.com.sdd.praapi.model.OperationDetail;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.Form;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

@SuppressWarnings({"unused", "FieldCanBeLocal"})
public class PraApiConnector implements RestSecurityManager {

    private final static Logger log = LogManager.getLogger(PraApiConnector.class);

    private final RestConnector restConnector;
    private final String GET_TOKEN_URL = "https://login.microsoftonline.com/933c9cbe-35d3-4416-abbd-ddd1bca5879c";
    private final String xApiKey;
    private final String clientId;
    private final String clientSecret;
    private final String scope;

    private final static int MAX_RETRIES = 3;
    private final static int RETRY_WAIT_MILIS=200;

    public PraApiConnector(PraApiConnectorContext context) {
        this.xApiKey = context.getPraApiXApiKey();
        this.clientId = context.getPraApiClientId();
        this.clientSecret = context.getPraApiClientSecret();
        this.scope = context.getPraApiScope();

        log.trace("Creando PraConnector para url [{}] con x-api-key [{}]", context.getPraApiBaseUrl(), xApiKey);

        RestConnectorEnvironment environment = new RestConnectorEnvironment(context.getPraApiBaseUrl());
        restConnector = new RestConnector(environment, this);

    }

    public RemittanceAdviceINResponse remittanceAdviceIN(RemittanceAdviceINRequest request) throws RestConnectorException {
        String path = "/jde/v2/remittanceAdvices/accountsReceivable";
        log.debug("[createPaymentRequest] Request POST procesar crear pago, request [{}] path [{}]", request, path);

        RestConnectorException lastException = null;
        for(int retry=MAX_RETRIES; retry > 0;retry--) {
            try {
                return restConnector.genericPost(request, RemittanceAdviceINResponse.class, CommonError.class, path);
            } catch (RestConnectorException e) {
                lastException = e;
                if (e.getStatus() == Response.Status.UNAUTHORIZED.getStatusCode()) {
                    getAccessToken(true);
                }
            }
            ThreadUtil.sleep(RETRY_WAIT_MILIS);
        }
        throw lastException;
    }

    public String remittanceAdviceOUT(String correlationId) throws RestConnectorException {
        String path = "/jde/v2/remittanceAdvices/accountsReceivable/" + correlationId;
        log.debug("[remittanceAdviceOUT] Request GET obtener Remittance Advice con path [{}]", path);

        RestConnectorException lastException = null;
        for(int retry=MAX_RETRIES; retry > 0;retry--) {
            try {
                return restConnector.genericGet(null, String.class, CommonError.class, path);
            } catch (RestConnectorException e) {
                lastException = e;
                if (e.getStatus() == Response.Status.UNAUTHORIZED.getStatusCode()) {
                    getAccessToken(true);
                }
            }
            ThreadUtil.sleep(RETRY_WAIT_MILIS);
        }
        throw lastException;
    }

    public SalesInvoiceINResponse salesInvoiceIN(SalesInvoiceINRequest request) throws RestConnectorException {
        String path = "/jde/v2/salesInvoices";
        log.debug("[salesInvoiceINResponse] Request POST procesar crear pago, request [{}] path [{}]", request, path);

        RestConnectorException lastException = null;
        for(int retry=MAX_RETRIES; retry > 0;retry--) {
            try {
                return restConnector.genericPost(request, SalesInvoiceINResponse.class, CommonError.class, path);
            } catch (RestConnectorException e) {
                lastException = e;
                if (e.getStatus() == Response.Status.UNAUTHORIZED.getStatusCode()) {
                    getAccessToken(true);
                }
            }
            try {
                Thread.sleep(RETRY_WAIT_MILIS);
            } catch (InterruptedException e) {
                // No hago nada;
            }
        }
        throw lastException;
    }

    public String salesInvoiceOUT(String correlationId) throws RestConnectorException {
        String path = "/jde/v2/salesInvoices/" + correlationId;
        log.debug("[salesInvoiceOUT] Request GET obtener Sales Invoice con path [{}]", path);

        RestConnectorException lastException = null;
        for(int retry=MAX_RETRIES; retry > 0;retry--) {
            try {
                return restConnector.genericGet(null, String.class, CommonError.class, path);
            } catch (RestConnectorException e) {
                lastException = e;
                if (e.getStatus() == Response.Status.UNAUTHORIZED.getStatusCode()) {
                    getAccessToken(true);
                }
            }
            ThreadUtil.sleep(RETRY_WAIT_MILIS);
        }
        throw lastException;
    }

    public String partinBuyerOUT(String correlationId) throws RestConnectorException {
        String path = "/jde/v2/partiesInformations/buyer/" + correlationId;
        log.debug("[partinBuyerOUT] Request GET obtener Party Information Buyer (PartinBuyer) con path [{}]", path);

        RestConnectorException lastException = null;
        for(int retry=MAX_RETRIES; retry > 0;retry--) {
            try {
                return restConnector.genericGet(null, String.class, CommonError.class, path);
            } catch (RestConnectorException e) {
                lastException = e;
                if (e.getStatus() == Response.Status.UNAUTHORIZED.getStatusCode()) {
                    getAccessToken(true);
                }
            }
            ThreadUtil.sleep(RETRY_WAIT_MILIS);
        }
        throw lastException;
    }

    public OperationDetail[] getAllPending() throws RestConnectorException {
        String path = "/common/v2/payloads/pending/correlationIDs";
        log.debug("[getAllPending] Request GET obtener operaciones pendientes, path [{}]", path);

        RestConnectorException lastException = null;
        for(int retry=MAX_RETRIES; retry > 0;retry--) {
            try {
                return restConnector.genericGet(null, OperationDetail[].class, CommonError.class, path);
            } catch (RestConnectorException e) {
                lastException = e;
                if (e.getStatus() == Response.Status.UNAUTHORIZED.getStatusCode()) {
                    getAccessToken(true);
                }
            }
            ThreadUtil.sleep(RETRY_WAIT_MILIS);
        }
        throw lastException;
    }

    public PayloadACKResponse[] payloadACK(String correlationId) throws RestConnectorException {
        String path = "/common/v2/payloads/collected/correlationIDs";
        log.debug("[payloadACK] Request POST payloadACK para el correlationID [{}] path [{}]", correlationId, path);
        RestConnectorException lastException = null;
        for(int retry=MAX_RETRIES; retry > 0;retry--) {
            try {
                return restConnector.genericPost(new String[]{correlationId}, PayloadACKResponse[].class, CommonError.class, path);
            } catch (RestConnectorException e) {
                lastException = e;
                if (e.getStatus() == Response.Status.UNAUTHORIZED.getStatusCode()) {
                    getAccessToken(true);
                }
            }
            ThreadUtil.sleep(RETRY_WAIT_MILIS);
        }
        throw lastException;
    }

    public String getCommercialAccountSummariesOUT(String correlationId) throws RestConnectorException {
        final String path = "/jde/v2/commercialAccountSummaries/accountsReceivable/" + correlationId;
        log.debug("[commercialAccountSummariesOUT] Request GET commercialAccountSummariesOUT para el correlationID [{}] path [{}]", correlationId, path);
        return restConnector.genericGet(null, String.class, CommonError.class, path);
    }

    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        if (builder != null) {
            builder.header("x-api-key", xApiKey);
            builder.header("Authorization", "Bearer " + getAccessToken(false));
        }
        return builder;
    }

    @Override
    public boolean getDisableHTTPSErrors() {
        return false;
    }

    @Override
    public boolean retryOnUnauthorized() {
        return false;
    }

    public String getAccessToken(boolean force) throws RestConnectorException {
        SimpleCache cache = SimpleCacheManager.getInstance().getCache(PraApiConnector.class.getName());
        String cacheKey = SimpleCacheManager.buildKey("accessToken");
        String accessToken = (String)cache.get(cacheKey);
        if (force || accessToken == null) {
            Form form = new Form();
            form.param("client_id", clientId);
            form.param("client_secret", clientSecret);
            form.param("scope", scope);
            form.param("grant_type", "client_credentials");

            log.debug("[getAccessToken] Por recuperar token con clientId [{}], clientSecret [{}] y scope [{}]", clientId, clientSecret, scope);

            RestConnectorEnvironment environment = new RestConnectorEnvironment();
            environment.baseUrl = GET_TOKEN_URL;
            RestConnector connector = new RestConnector(environment, null);
            TokenOAuth2 tokenOAuth2Response = connector.genericPost(form, TokenOAuth2.class, "/oauth2/v2.0/token", MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);

            log.debug("[getAccessToken] Token recuperado [{}]. Expira en [{}]segs. Lo agrego al a cache", tokenOAuth2Response, tokenOAuth2Response.getExpiresIn());
            accessToken = tokenOAuth2Response.getAccessToken();
            cache.put(cacheKey, accessToken, tokenOAuth2Response.getExpiresIn()-10);
        }

        return accessToken;
    }
}