package ar.com.sdd.praapi.core;

import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import ar.com.sdd.commons.rest.model.TokenOAuth2;
import ar.com.sdd.praapi.io.RemittanceAdviceINRequest;
import ar.com.sdd.praapi.io.RemittanceAdviceINResponse;
import ar.com.sdd.praapi.io.SalesInvoiceINRequest;
import ar.com.sdd.praapi.io.SalesInvoiceINResponse;
import ar.com.sdd.praapi.model.OperationDetail;
import ar.com.sdd.praapi.util.SimpleCache;
import org.apache.log4j.Logger;

import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.Form;
import javax.ws.rs.core.MediaType;
import java.time.temporal.ChronoUnit;

public class PraApiConnector implements RestSecurityManager {

    private final static Logger log = Logger.getLogger(PraApiConnector.class);

    private final RestConnector restConnector;
    private final String GET_TOKEN_URL = "https://login.microsoftonline.com/933c9cbe-35d3-4416-abbd-ddd1bca5879c";
    private final String xApiKey;
    private final String clientId;
    private final String clientSecret;
    private final String scope;

    private final SimpleCache<String, String> cache;

    public PraApiConnector(PraApiConnectorContext context) {
        this.xApiKey = context.getPraApiXApiKey();
        this.clientId = context.getPraApiClientId();
        this.clientSecret = context.getPraApiClientSecret();
        this.scope = context.getPraApiScope();

        log.debug("Creando PraConnector para url [" + context.getPraApiBaseUrl() + "] con x-api-key [" + xApiKey + "]");

        RestConnectorEnvironment environment = new RestConnectorEnvironment(context.getPraApiBaseUrl());
        restConnector = new RestConnector(environment, this);

        // Inicializo la cache que va a contener el accessToken. El mismo expira luego de 1 hora
        cache = SimpleCache.builder()
                .maximumSize(1)                             // Puedo tener solo 1 token activo
                .expireAfter(1, ChronoUnit.HOURS) // El token expira luego de 1 hora
                .build();
    }

    public RemittanceAdviceINResponse remittanceAdviceIN(RemittanceAdviceINRequest request) throws RestConnectorException {
        String path = "/jde/v2/remittanceAdvices/accountsReceivable";
        log.debug("[createPaymentRequest] Request POST procesar crear pago, request [" + request + "] path [" + path + "]");
        return restConnector.genericPost(request, RemittanceAdviceINResponse.class, path);
    }

    public String remittanceAdviceOUT(String correlationId) throws RestConnectorException {
        String path = "/jde/v2/remittanceAdvices/accountsReceivable/" + correlationId;
        log.debug("[remittanceAdviceOUT] Request GET obtener Remittance Advice con path [" + path + "]");
        return restConnector.genericGet(null, String.class, path);
    }

    public SalesInvoiceINResponse salesInvoiceIN(SalesInvoiceINRequest request) throws RestConnectorException {
        String path = "/jde/v2/salesInvoices";
        log.debug("[salesInvoiceINResponse] Request POST procesar crear pago, request [" + request + "] path [" + path + "]");
        return restConnector.genericPost(request, SalesInvoiceINResponse.class, path);
    }

    public String salesInvoiceOUT(String correlationId) throws RestConnectorException {
        String path = "/jde/v2/salesInvoices/" + correlationId;
        log.debug("[salesInvoiceOUT] Request GET obtener Sales Invoice con path [" + path + "]");
        return restConnector.genericGet(null, String.class, path);
    }

    public String partinBuyerOUT(String correlationId) throws RestConnectorException {
        String path = "/jde/v2/partiesInformations/buyer/" + correlationId;
        log.debug("[partinBuyerOUT] Request GET obtener Party Information Buyer (PartinBuyer) con path [" + path + "]");
        return restConnector.genericGet(null, String.class, path);
    }

    public OperationDetail[] getAllPending() throws RestConnectorException {
        String path = "/common/v2/payloads/pending/correlationIDs";
        log.debug("[getAllPending] Request GET obtener operaciones pendientes, path [" + path + "]");
        return restConnector.genericGet(null, OperationDetail[].class, path);
    }

    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        if (builder != null) {
            builder.header("x-api-key", xApiKey);
            builder.header("Authorization", "Bearer " + getAccessToken());
        }
        return builder;
    }

    @Override
    public boolean getDisableHTTPSErrors() {
        return false;
    }

    public String getAccessToken() throws RestConnectorException {
        String accessToken = cache.getIfPresent("accessToken");
        if (accessToken == null) {
            Form form = new Form();
            form.param("client_id", clientId);
            form.param("client_secret", clientSecret);
            form.param("scope", scope);
            form.param("grant_type", "client_credentials");

            log.debug("[getAccessToken] Por recuperar token con clientId [" + clientId + "], clientSecret [" + clientSecret + "] y scope [" + scope + "]");

            RestConnectorEnvironment environment = new RestConnectorEnvironment();
            environment.baseUrl = GET_TOKEN_URL;
            RestConnector connector = new RestConnector(environment, null);
            TokenOAuth2 tokenOAuth2Response = connector.genericPost(form, TokenOAuth2.class, "/oauth2/v2.0/token", MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);

            log.debug("[getAccessToken] Token recuperado [" + tokenOAuth2Response + "]. Lo agrego al a cache");
            accessToken = tokenOAuth2Response.getAccessToken();
            cache.put("accessToken", accessToken);
        }

        return accessToken;
    }
}