package ar.com.sdd.tasa.core;

import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import ar.com.sdd.commons.rest.model.TokenOAuth2;
import ar.com.sdd.commons.rest.util.QueryBuilder;
import ar.com.sdd.tasa.entity.InvoiceListItem;
import ar.com.sdd.tasa.entity.InvoiceListItems;
import org.apache.log4j.Logger;

import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.Form;
import javax.ws.rs.core.MediaType;
import java.util.Date;

public class TasaConnector implements RestSecurityManager {

    private static final Logger log = Logger.getLogger(TasaConnector.class);

    private RestConnector restConnector;
    private String clientId;
    private String clientSecret;

    public TasaConnector(TasaConnectorContext tasaConnectorContext) {
        this(tasaConnectorContext.getTasaBaseUrl(), tasaConnectorContext.getTasaClientId(), tasaConnectorContext.getTasaClientSecret());
    }

    public TasaConnector(String baseUrl, String clientId, String clientSecret) {
        log.debug("Creando connector con baseUrl [" + baseUrl + "], clientId [" + clientId + "], clientSecret [" + clientSecret + "]");
        this.clientId = clientId;
        this.clientSecret = clientSecret;
        RestConnectorEnvironment environment = new RestConnectorEnvironment(baseUrl, null, null, null, null);
        restConnector = new RestConnector(environment, this);
    }

    public InvoiceListItems getInvoices(String cuit, InvoiceListItem.LineOfBusiness lineOfBusiness, Date fromDueDate, Date toDueDate) throws RestConnectorException {
        QueryBuilder queryBuilder = QueryBuilder.Builder().path("/billingmanagement/eerr/v1/invoices")
                .add("tipoDocumento", "CUIT")
                .add("numeroDocumento", cuit)
                .add("lineOfBusiness", lineOfBusiness.name());

        if (fromDueDate != null) {
            queryBuilder.add("fromDueDate", fromDueDate);
        }

        if (toDueDate != null) {
            queryBuilder.add("toDueDate", toDueDate);
        }

        String path = queryBuilder.build();

        log.debug("[getInvoices] Request GET invoices con cuit [" + cuit + "], lineOfBusiness [" + lineOfBusiness + "], fromDueDate [" + fromDueDate + "], toDueDate [" + toDueDate + "], path [" + path + "]");
        InvoiceListItems response = restConnector.genericGet(null, InvoiceListItems.class, path);
        log.debug("[getInvoices] Response GET invoices [" + response + "]");
        return response;
    }

    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        TokenOAuth2 token = getToken();
        log.trace("[addHeaders] Token recuperado [" + token + "]");
        builder.header("authorization", "Bearer " + token.getAccessToken());
        builder.header("x-ibm-client-id", clientId);
        return builder;
    }

    private TokenOAuth2 getToken() throws RestConnectorException {
        Form form = new Form();
        form.param("client_id", clientId);
        form.param("client_secret", clientSecret);
        form.param("grant_type", "client_credentials");

        log.debug("[getToken] Por recuperar token con clientId [" + clientId + "], clientSecret [" + clientSecret + "]");

        TokenOAuth2 tokenOAuth2Response = restConnector.genericPost(form, TokenOAuth2.class, "/v1/oauth2/token", MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);

        log.debug("[getToken] Token recuperado [" + tokenOAuth2Response + "]");
        return tokenOAuth2Response;
    }

}
