package ar.com.sdd.tasa.core;

import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import ar.com.sdd.commons.rest.model.TokenOAuth2;
import ar.com.sdd.commons.rest.util.QueryBuilder;
import ar.com.sdd.tasa.entity.InvoiceListItem;
import ar.com.sdd.tasa.io.PaymentRequest;
import ar.com.sdd.tasa.io.PaymentResponse;
import org.apache.log4j.Logger;

import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.Form;
import javax.ws.rs.core.MediaType;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;

public class TasaConnector implements RestSecurityManager {

    private static final Logger log = Logger.getLogger(TasaConnector.class);

    public static final SimpleDateFormat sdfTasa = new SimpleDateFormat("yyyyMMdd");

    private RestConnectorEnvironment environment;
    private RestConnector restConnector;
    private String clientId;
    private String clientSecret;
    private String scope;

    public TasaConnector(TasaConnectorContext tasaConnectorContext) {
        this(tasaConnectorContext.getTasaBaseUrl(), tasaConnectorContext.getTasaClientId(), tasaConnectorContext.getTasaClientSecret(), tasaConnectorContext.getTasaScope(),
                tasaConnectorContext.getTargetNameFijo(), tasaConnectorContext.getTargetNameMovil(),
                tasaConnectorContext.getImportTaskFijo(), tasaConnectorContext.getImportTaskMovil());
    }

    public TasaConnector(String baseUrl, String clientId, String clientSecret, String scope, String targetNameFijo, String targetNameMovil,
                         String importTaskFijo, String importTaskMovil) {
        log.debug("Creando connector Tasa con baseUrl [" + baseUrl + "], clientId [" + clientId + "], clientSecret [" + clientSecret + "], scope [" + scope + "], " +
                "targetNameFijo [" + targetNameFijo + "], targetNameMovil [" + targetNameMovil + "] " +
                "importTaskFijo [" + importTaskFijo + "], importTaskMovil [" + importTaskMovil + "]");
        this.scope = scope;
        this.clientId = clientId;
        this.clientSecret = clientSecret;

        environment = new RestConnectorEnvironment(baseUrl, null, null, null, null);
        restConnector = new RestConnector(environment, this);
    }

    public InvoiceListItem[] getInvoices(String cuit, TasaConst.LineOfBusiness lineOfBusiness, Date fromDueDate, Date toDueDate) throws RestConnectorException {
        QueryBuilder queryBuilder = QueryBuilder.Builder().path("/billingmanagement/eerr/v1/invoices")
                .add("tipoDocumento", "CUIT")
                .add("numeroDocumento", cuit)
                .add("lineOfBusiness", lineOfBusiness.name());

        if (fromDueDate != null) {
            queryBuilder.add("fromDueDate", sdfTasa.format(fromDueDate));
        }

        if (toDueDate != null) {
            queryBuilder.add("toDueDate", sdfTasa.format(toDueDate));
        }

        String path = queryBuilder.build();

        log.debug("[getInvoices] Request GET invoices con cuit [" + cuit + "], lineOfBusiness [" + lineOfBusiness + "], fromDueDate [" + fromDueDate + "], toDueDate [" + toDueDate + "], path [" + path + "]");
        InvoiceListItem[] response = restConnector.genericGet(null,InvoiceListItem[].class, path);
        log.debug("[getInvoices] Response GET invoices [" + Arrays.toString(response) + "]");
        return response;
    }

    public PaymentResponse postPayment(PaymentRequest paymentRequest) throws RestConnectorException {
        log.debug("[postPayment] Request POST payments con paymentRequest [" + paymentRequest + "]");
        PaymentResponse response = restConnector.genericPost(paymentRequest, PaymentResponse.class, "/paymentmanagement/v1/payment");
        log.debug("[postPayment] Response POST payments [" + response + "]");
        return response;
    }

    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        TokenOAuth2 token = getToken();
        log.trace("[addHeaders] Token recuperado [" + token + "]");
        builder.header("authorization", "Bearer " + token.getAccessToken());
        builder.header("x-ibm-client-id", clientId);
        return builder;
    }

    private TokenOAuth2 getToken() throws RestConnectorException {
        Form form = new Form();
        form.param("client_id", clientId);
        form.param("client_secret", clientSecret);
        form.param("grant_type", "client_credentials");
        form.param("scope", scope);

        log.debug("[getToken] Por recuperar token con clientId [" + clientId + "], clientSecret [" + clientSecret + "], scope [" + scope + "]");

        //Creo un nuevo conector solo apra el token
        RestConnector tokenConnector = new RestConnector(environment, null);
        TokenOAuth2 tokenOAuth2Response = tokenConnector.genericPost(form, TokenOAuth2.class, "/v1/oauth2/token", MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);

        log.debug("[getToken] Token recuperado [" + tokenOAuth2Response + "]");
        return tokenOAuth2Response;
    }

}
