package ar.com.sdd.teco.shiva.core

import ar.com.sdd.commons.rest.core.RestConnector
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment
import ar.com.sdd.commons.rest.core.RestConnectorException
import ar.com.sdd.commons.rest.core.RestSecurityManager
import ar.com.sdd.commons.rest.util.QueryBuilder
import ar.com.sdd.commons.rest.util.RestConnectorUtil
import ar.com.sdd.teco.shiva.io.*
import org.apache.logging.log4j.LogManager
import org.apache.logging.log4j.Logger
import java.util.*
import javax.ws.rs.client.Invocation
import javax.ws.rs.core.Form
import javax.ws.rs.core.MediaType

open class TecoShivaConnector(context: TecoShivaConnectorContext) : RestSecurityManager {

    private val log: Logger = LogManager.getLogger(TecoShivaConnector::class.java)

    private val restConnector: RestConnector
    private val context: TecoShivaConnectorContext

    init {
        log.debug("Creando TecoShivaConnector para authUrl [${context.getAuthUrl()}], endpoint [${context.getEndpointUrl()}], user [${context.getUser()}], pass [${context.getPass()}]")
        this.context = context
        val environment = RestConnectorEnvironment(context.getEndpointUrl())
        restConnector = RestConnector(environment, this)
    }

    // Implemento una minicache del token actual
    companion object {
        var currentToken: TokenOAuth2Shiva? = null;
        var currentTokenExpires: Date = Date();
        fun cacheResetToken() {
            currentToken = null;
            currentTokenExpires = Date();
        };
        fun cacheTokenIsExpired(): Boolean {
            return currentToken==null || currentTokenExpires.before(Date())
        }
        fun cacheSetToken(token: TokenOAuth2Shiva) {
            currentToken = token
            currentTokenExpires = Date(currentToken!!.expiresIn * 1000 + System.currentTimeMillis()- 3600*1000) //la expiracion - 1 hora
        }
        fun cacheGetToken(): String {
            return currentToken!!.idToken
        }
    }
    // Opcion para resetear el token
    fun resetToken() {
        cacheResetToken();
    }

    @Throws(RestConnectorException::class)
    override fun addHeaders(builder: Invocation.Builder?): Invocation.Builder? {
        //Obtengo el token de la cache, si esta vencido lo renuevo
        if (cacheTokenIsExpired()) {
            //Recupero el token
            log.debug("Recuperando Token")
            val form = Form()
            form.param("grant_type", "client_credentials")

            val environment = RestConnectorEnvironment()
            environment.baseUrl = context.getAuthUrl()
            val connector = RestConnector(environment, object : RestSecurityManager {
                override fun addHeaders(builder: Invocation.Builder): Invocation.Builder {
                    builder.header("Authorization", RestConnectorUtil.getBasicAuthHeader(context.getUser(), context.getPass()))
                    return builder
                }

                override fun getDisableHTTPSErrors(): Boolean {
                    return false
                }
            })

            val tokenOAuth2Response = connector.genericPost(form, TokenOAuth2Shiva::class.java, "", MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED)
            log.debug("Token recuperado [$tokenOAuth2Response]")
            cacheSetToken(tokenOAuth2Response)
        }

        val idToken = cacheGetToken()
        builder?.header("Authorization", "Bearer $idToken")

        return builder
    }

    override fun getDisableHTTPSErrors(): Boolean {
        return false;
    }

    @Throws(RestConnectorException::class)
    fun consultaCliente(tipoDocumento: String, numeroDocumento: String): ConsultaClienteTecoShivaResponse {
        val queryBuilder: QueryBuilder = QueryBuilder.Builder().path("/cliente")
                .add("tipoDocumento", tipoDocumento)
                .add("numeroDocumento", numeroDocumento)

        val path = queryBuilder.build()
        log.debug("[consultaClientes] Request GET cliente con tipoDocumento [$tipoDocumento], numeroDocumento [$numeroDocumento], path [$path]")
        val response = restConnector.genericGet(null, ConsultaClienteTecoShivaResponse::class.java, TecoShivaErrorResponse::class.java, path)
        log.debug("[consultaClientes] Response GET cliente [$response]")

        TecoShivaConnector(context).resetToken();
        return response
    }



    @Throws(RestConnectorException::class)
    fun consultaDeuda(request: ConsultaDeudaTecoShivaRequest): ConsultaDeudaTecoShivaResponse {
        val path = "/deuda"
        log.debug("[consultaDeuda] Request POST deuda con request [$request], path [$path]")
        val response = restConnector.genericPost(request, ConsultaDeudaTecoShivaResponse::class.java, TecoShivaErrorResponse::class.java, path)
        log.debug("[consultaDeuda] Response POST deuda [$response]")

        return response
    }

}