package ar.com.sdd.teco.shiva.core

import ar.com.sdd.commons.rest.core.RestConnector
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment
import ar.com.sdd.commons.rest.core.RestConnectorException
import ar.com.sdd.commons.rest.core.RestSecurityManager
import ar.com.sdd.commons.rest.util.QueryBuilder
import ar.com.sdd.commons.rest.util.RestConnectorUtil
import ar.com.sdd.commons.util.SimpleCache
import ar.com.sdd.commons.util.SimpleCacheManager
import ar.com.sdd.teco.shiva.io.*
import org.apache.logging.log4j.LogManager
import org.apache.logging.log4j.Logger
import java.util.*
import javax.ws.rs.client.Invocation
import javax.ws.rs.core.Form
import javax.ws.rs.core.MediaType


open class TecoShivaConnector(context: TecoShivaConnectorContext) : RestSecurityManager {

    private val log: Logger = LogManager.getLogger(TecoShivaConnector::class.java)

    private val restConnector: RestConnector
    private val context: TecoShivaConnectorContext

    init {
        log.debug("Creando TecoShivaConnector para authUrl [${context.getAuthUrl()}], endpoint [${context.getEndpointUrl()}], user [${context.getUser()}]")
        this.context = context
        val environment = RestConnectorEnvironment(context.getEndpointUrl())
        restConnector = RestConnector(environment, this)
    }

    @Throws(RestConnectorException::class)
    override fun addHeaders(builder: Invocation.Builder?): Invocation.Builder? {
        val idToken = getAccessToken(false)
        builder?.header("Authorization", "Bearer $idToken")
        return builder
    }

    fun getAccessToken(force:Boolean): String {
        val cache: SimpleCache = SimpleCacheManager.getInstance().getCache(TecoShivaConnector::class.java.getName())
        val cacheKey = SimpleCacheManager.buildKey("accessToken")
        var accessToken = cache.get(cacheKey)
        if (force || accessToken == null) {
            log.debug("Recuperando Token.")
            val form = Form()
            form.param("grant_type", "client_credentials")

            val environment = RestConnectorEnvironment()
            environment.baseUrl = context.getAuthUrl()
            val connector = RestConnector(environment, object : RestSecurityManager {
                override fun addHeaders(builder: Invocation.Builder): Invocation.Builder {
                    builder.header("Authorization", RestConnectorUtil.getBasicAuthHeader(context.getUser(), context.getPass()))
                    return builder
                }

                override fun retryOnUnauthorized(): Boolean {
                    return false
                }

                override fun getDisableHTTPSErrors(): Boolean {
                    return false
                }
            })

            val tokenOAuth2Response = connector.genericPost(form, TokenOAuth2Shiva::class.java, "", MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED)
            log.debug("Token recuperado [$tokenOAuth2Response]")
            accessToken = tokenOAuth2Response.idToken
            cache.put(cacheKey, accessToken, tokenOAuth2Response.getExpiresIn())
        }
        return accessToken as String;
    }

    override fun retryOnUnauthorized(): Boolean {
        return false; // Porque no tengo un 401 sino un 403 con "Acceso Denegado"
    }

    override fun getDisableHTTPSErrors(): Boolean {
        return false
    }

    @Throws(RestConnectorException::class)
    fun consultaCliente(tipoDocumento: String, numeroDocumento: String): ConsultaClienteTecoShivaResponse {
        val queryBuilder: QueryBuilder = QueryBuilder.Builder().path("/cliente")
                .add("tipoDocumento", tipoDocumento)
                .add("numeroDocumento", numeroDocumento)

        val path = queryBuilder.build()
        log.debug("[consultaClientes] Request GET cliente con tipoDocumento [$tipoDocumento], numeroDocumento [$numeroDocumento], path [$path]")

        try {
            val response = restConnector.genericGet(null, ConsultaClienteTecoShivaResponse::class.java, TecoShivaErrorResponse::class.java, path)
            log.debug("[consultaClientes] Response GET cliente [$response]")
            return response
        } catch (restException:RestConnectorException ) {
            if (restException.getEntityError() != null) {
                val errorResponse : TecoShivaErrorResponse = restException.getEntityError () as TecoShivaErrorResponse;
                if (errorResponse.mensaje?.contains("Acceso denegado") == true) {
                    getAccessToken(true)
                    val response = restConnector.genericGet(null, ConsultaClienteTecoShivaResponse::class.java, TecoShivaErrorResponse::class.java, path)
                    log.debug("[consultaClientes] Response GET cliente [$response]")
                    return response
                }
            }
            throw  restException //Si llegue aca era otra cosa
        }
    }



    @Throws(RestConnectorException::class)
    fun consultaDeuda(request: ConsultaDeudaTecoShivaRequest): ConsultaDeudaTecoShivaResponse {
        val path = "/deuda"
        log.debug("[consultaDeuda] Request POST deuda con request [$request], path [$path]")
        try {
            val response = restConnector.genericPost(request, ConsultaDeudaTecoShivaResponse::class.java, TecoShivaErrorResponse::class.java, path)
            log.debug("[consultaDeuda] Response POST deuda [$response]")
            return response
        } catch (restException:RestConnectorException ) {
            if (restException.getEntityError() != null) {
                val errorResponse : TecoShivaErrorResponse = restException.getEntityError () as TecoShivaErrorResponse;
                if (errorResponse.mensaje?.contains("Acceso denegado") == true) {
                    getAccessToken(true)
                    val response = restConnector.genericPost(request, ConsultaDeudaTecoShivaResponse::class.java, TecoShivaErrorResponse::class.java, path)
                    log.debug("[consultaDeuda] Response POST deuda [$response]")
                    return response
                }
            }
            throw  restException //Si llegue aca era otra cosa
        }
    }

}