package ar.com.sdd.teco.shiva.core

import ar.com.sdd.commons.rest.core.RestConnector
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment
import ar.com.sdd.commons.rest.core.RestConnectorException
import ar.com.sdd.commons.rest.core.RestSecurityManager
import ar.com.sdd.commons.rest.util.QueryBuilder
import ar.com.sdd.commons.rest.util.RestConnectorUtil
import ar.com.sdd.commons.util.SimpleCache
import ar.com.sdd.commons.util.SimpleCacheManager
import ar.com.sdd.commons.util.StringUtil
import ar.com.sdd.teco.shiva.io.*
import org.apache.logging.log4j.LogManager
import org.apache.logging.log4j.Logger
import java.util.*
import javax.ws.rs.client.Invocation
import javax.ws.rs.core.Form
import javax.ws.rs.core.MediaType


open class TecoShivaConnector(context: TecoShivaConnectorContext) : RestSecurityManager {

    private val log: Logger = LogManager.getLogger(TecoShivaConnector::class.java)

    private val restConnector: RestConnector
    private val context: TecoShivaConnectorContext

    init {
        log.debug("Creando TecoShivaConnector para authUrl [${context.getAuthUrl()}], endpoint [${context.getEndpointUrl()}], user [${context.getUser()}]")
        this.context = context
        val environment = RestConnectorEnvironment(context.getEndpointUrl())
        restConnector = RestConnector(environment, this)
    }

    @Throws(RestConnectorException::class)
    override fun addHeaders(builder: Invocation.Builder?): Invocation.Builder? {
        val idToken = getAccessToken(false)
        builder?.header("Authorization", "Bearer $idToken")
        return builder
    }


    /*
    El basic es client_id:client_secret pasado por base 64
    curl 'https://idpsesiont.telecom.com.ar/openam/oauth2/authserver/access_token'
                 -H 'Content-Type: application/x-www-form-urlencoded'
                 -H 'Authorization: Basic ZWJlZDc1MzYtNTU3OC00YjE3LTg0ZmItYzg4M2M5ZjVmOGZiOk1HdTFSdFl0T1JmcWJ5SVpMX0pSeUI2MHlCamlvZzRVeU40UkdSMHh4ZjZtTU96SkhkMFloXzd5UkRYdG1USjNUTkdkRVRubWV3X2Y4aTluXzAyTUZR'
                 -H 'Cookie: TS01d0e0ea=0154ce2499d6a07cacc2ba56fe6c4dc24f700940c4c3f19f3882ef844fc40eecc2edb18ba22f54700f021d04123c8a4247e1ab88d5'
                 --data-urlencode 'grant_type=client_credentials'                 -\-data-urlencode  'scope=openid'

    { "access_token":"Gx_AmV_uefepkGJQM_exDIk4w0A",
      "scope":"openid",
      "id_token":"eyJ0eXAiOiJKV1QiLCJraWQiOiJ3VTNpZklJYUxPVUFSZVJCL0ZHNmVNMVAxUU09IiwiYWxnIjoiUlMyNTYifQ.eyJhdF9oYXNoIjoiMjljMEZwZDdmUFNfang1WnBNdUVNUSIsInN1YiI6ImViZWQ3NTM2LTU1NzgtNGIxNy04NGZiLWM4ODNjOWY1ZjhmYiIsImF1ZGl0VHJhY2tpbmdJZCI6ImI0MTcxNDZlLTI4NzgtNDc3OC05YzBhLTM1N2ZhZmYzNzJmMi0xOTAyNzc4IiwiaXNzIjoiaHR0cHM6Ly9pZHBzZXNpb250LnRlbGVjb20uY29tLmFyOjQ0My9vcGVuYW0vb2F1dGgyL2F1dGhzZXJ2ZXIiLCJ0b2tlbk5hbWUiOiJpZF90b2tlbiIsImF1ZCI6ImViZWQ3NTM2LTU1NzgtNGIxNy04NGZiLWM4ODNjOWY1ZjhmYiIsImF6cCI6ImViZWQ3NTM2LTU1NzgtNGIxNy04NGZiLWM4ODNjOWY1ZjhmYiIsImF1dGhfdGltZSI6MTc2NDAxOTAwMCwicmVhbG0iOiIvYXV0aHNlcnZlciIsImV4cCI6MTc2NDAzMzQwMCwidG9rZW5UeXBlIjoiSldUVG9rZW4iLCJpYXQiOjE3NjQwMTkwMDB9.fDVFEWy0tG0DkaAS4ltFYSayeD4HRggZ8PaBE8WmA27llRD8UFB-mMIohxuWP9mPw2lwTgsHEd_MeVnT_UmCFjDVbRBz1VK7SgZg4k1Cv5JaBrwzGhTDHpzhA4FQbb4-XGQy8GxATeKw7GibNC1YTidNpj0RVWfS-HpiuooiuXv1YlNZGo-joKpwOrBFOmX7xqigffc_vQxw9IHOHZO2UhHZxUjdIAzbAvkt3kzxiKv4DZeDofcAw46UYJ3ewBtKhAzYgjEJzq8e4Nzo2XRdCRv6WZNJdu9Ynum2sdRm074K3-edyONzbmFa3GbeSrieY3CWtr0CCqa4utDfqNdkvA",
      "token_type":"Bearer",
      "expires_in":14399}
    */
    fun getAccessToken(force:Boolean): String {
        val cache: SimpleCache = SimpleCacheManager.getInstance().getCache(TecoShivaConnector::class.java.getName())
        val cacheKey = SimpleCacheManager.buildKey("accessToken")
        var accessToken = cache.get(cacheKey)
        if (force || accessToken == null) {
            log.debug("Recuperando Token.")
            val form = Form()
            form.param("grant_type", "client_credentials")
            form.param("scope", "openid") //Lo agrego, no se si en el modelo viejo molesta

            val environment = RestConnectorEnvironment()
            environment.baseUrl = context.getAuthUrl()
            val connector = RestConnector(environment, object : RestSecurityManager {
                override fun addHeaders(builder: Invocation.Builder): Invocation.Builder {
                    if (!StringUtil.isEmpty(context.getCookie())) {
                        builder.header("Cookie", context.getCookie())
                    }
                    builder.header("Authorization", RestConnectorUtil.getBasicAuthHeader(context.getUser(), context.getPass()))
                    return builder
                }

                override fun retryOnUnauthorized(): Boolean {
                    return false
                }

                override fun getDisableHTTPSErrors(): Boolean {
                    return true
                }
            })

            val tokenOAuth2Response = connector.genericPost(form, TokenOAuth2Shiva::class.java, "", MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED)
            log.debug("Token recuperado [$tokenOAuth2Response]")
            accessToken = tokenOAuth2Response.idToken
            cache.put(cacheKey, accessToken, tokenOAuth2Response.getExpiresIn())
        }
        return accessToken as String
    }

    override fun retryOnUnauthorized(): Boolean {
        return false // Porque no tengo un 401 sino un 403 con "Acceso Denegado"
    }

    override fun getDisableHTTPSErrors(): Boolean {
        return true
    }

    @Throws(RestConnectorException::class)
    fun consultaCliente(tipoDocumento: String, numeroDocumento: String): ConsultaClienteTecoShivaResponse {
        val queryBuilder: QueryBuilder = QueryBuilder.Builder().path("/cliente")
                .add("tipoDocumento", tipoDocumento)
                .add("numeroDocumento", numeroDocumento)

        val path = queryBuilder.build()
        log.debug("[consultaClientes] Request GET cliente con tipoDocumento [$tipoDocumento], numeroDocumento [$numeroDocumento], path [$path]")

        try {
            val response = restConnector.genericGet(null, ConsultaClienteTecoShivaResponse::class.java, TecoShivaErrorResponse::class.java, path)
            log.debug("[consultaClientes] Response GET cliente [$response]")
            return response
        } catch (restException:RestConnectorException ) {
            if (restException.getEntityError() != null) {
                val errorResponse : TecoShivaErrorResponse = restException.getEntityError () as TecoShivaErrorResponse
                if (errorResponse.mensaje?.contains("Acceso denegado") == true) {
                    getAccessToken(true)
                    val response = restConnector.genericGet(null, ConsultaClienteTecoShivaResponse::class.java, TecoShivaErrorResponse::class.java, path)
                    log.debug("[consultaClientes] Response GET cliente 2 [$response]")
                    return response
                }
            }
            throw  restException //Si llegue aca era otra cosa
        }
    }



    @Throws(RestConnectorException::class)
    fun consultaDeuda(request: ConsultaDeudaTecoShivaRequest): ConsultaDeudaTecoShivaResponse {
        val path = "/deuda"
        log.debug("[consultaDeuda] Request POST deuda con request [$request], path [$path]")
        try {
            val response = restConnector.genericPost(request, ConsultaDeudaTecoShivaResponse::class.java, TecoShivaErrorResponse::class.java, path)
            log.debug("[consultaDeuda] Response POST deuda [{}]", response)
            return response
        } catch (restException:RestConnectorException ) {
            if (restException.entityError != null) {
                val errorResponse : TecoShivaErrorResponse = restException.entityError as TecoShivaErrorResponse
                if (errorResponse.mensaje?.contains("Acceso denegado") == true) {
                    getAccessToken(true)
                    val response = restConnector.genericPost(request, ConsultaDeudaTecoShivaResponse::class.java, TecoShivaErrorResponse::class.java, path)
                    log.debug("[consultaDeuda] Response POST deuda 2 [$response]")
                    return response
                }
            }
            throw  restException //Si llegue aca era otra cosa
        }
    }

}